# CLAUDE.md

This file provides guidance to Claude Code (claude.ai/code) when working with code in this repository.

## Repository Overview

This is a **PHP client library for the Unleashed API**, automatically generated using OpenAPI Generator from the OpenAPI specification in `swagger/swagger.json`. The repository provides a comprehensive PHP SDK to interact with Unleashed Software's inventory management system.

**Critical**: This is an auto-generated codebase. Most PHP code in `lib/`, `docs/`, and `test/` should NOT be edited directly as it will be regenerated when the OpenAPI specification changes.

## Code Generation Workflow

### Regenerating the Client Library

When the OpenAPI specification changes, regenerate the entire client library:

```bash
cd swagger
./build.sh
```

This script:
1. Removes existing `lib/`, `docs/`, and `test/` directories
2. Runs OpenAPI Generator via Docker using `swagger/swagger.json` as input
3. Uses configuration from `swagger/config.json`
4. Generates PHP 8.0+ client code with namespace `Unleashed`

**Important**: The `composer.json` file is protected from regeneration via `.openapi-generator-ignore`.

### Validating the OpenAPI Specification

Before regenerating, validate the OpenAPI spec:

```bash
npx --yes @apidevtools/swagger-cli validate swagger/swagger.json
```

The swagger.json file must be valid OpenAPI 2.0 (Swagger) format.

## Architecture

### Directory Structure

- **`swagger/swagger.json`** - OpenAPI 2.0 specification defining the Unleashed API
- **`swagger/config.json`** - OpenAPI Generator configuration
- **`swagger/build.sh`** - Script to regenerate the entire client library
- **`lib/Api/`** - Auto-generated API client classes (one per resource)
- **`lib/Model/`** - Auto-generated model/entity classes
- **`src/`** - Custom code that persists across regenerations
  - `HeadersMiddleware.php` - Custom Guzzle middleware for API authentication
- **`test/`** - Auto-generated PHPUnit tests
- **`docs/`** - Auto-generated API documentation

### API Client Pattern

All API clients follow the same pattern:
- Located in `lib/Api/` (e.g., `PurchaseOrdersApi.php`, `ProductsApi.php`)
- Each API class provides methods for HTTP operations (GET, POST, PUT, DELETE)
- Methods accept parameters and return model objects or arrays
- Uses Guzzle HTTP client under the hood

### Authentication

The Unleashed API requires custom headers for authentication. The custom middleware in `src/HeadersMiddleware.php` handles this and is not regenerated.

## Development Commands

### Running Tests

```bash
composer install
vendor/bin/phpunit
```

### Code Style

The repository includes PHP-CS-Fixer configuration:

```bash
# Check code style
vendor/bin/php-cs-fixer fix --dry-run --diff

# Fix code style
vendor/bin/php-cs-fixer fix
```

## Working with OpenAPI Definitions

### Modifying the API Specification

When adding or updating endpoints in `swagger/swagger.json`:

1. **Use definition references** instead of inline schemas:
   ```json
   {
     "schema": {
       "$ref": "#/definitions/purchaseOrder"
     }
   }
   ```
   Not:
   ```json
   {
     "schema": {
       "type": "object",
       "properties": { ... }
     }
   }
   ```

2. **Follow existing patterns** - Look at similar endpoints like `/SalesOrders` when adding new resources

3. **Add proper tags** - Group endpoints using tags (e.g., "Purchase Orders", "Products")

4. **Include query parameters** - Use `$ref: "#/parameters/..."` for common parameters like `pageSize`, `pageNumber`

5. **Date format** - All dates must be in format `yyyy-MM-ddTHH:mm:ss.fff` (ISO 8601 with milliseconds)

### Common OpenAPI Patterns in This Spec

- **Pagination**: Most list endpoints have a `/Page/{pageNumber}` variant
- **Filtering**: List endpoints accept query parameters for filtering
- **GUID identifiers**: Path parameters use `{id}` expecting GUID format
- **Paged responses**: Use `pagedXXX` definitions that include `Pagination` and `Items` array

## API Reference

Base URL: `https://api.unleashedsoftware.com`

Key API resources:
- **Products** - Product catalog management
- **Purchase Orders** - Purchase order lifecycle
- **Sales Orders** - Sales order management
- **Customers** - Customer records
- **Suppliers** - Supplier information
- **Stock On Hand** - Inventory levels
- **Warehouses** - Warehouse management
- **Assemblies** - Assembly/kit management

## Package Details

- **Namespace**: `Unleashed`
- **Autoloading**: PSR-4 autoloading for both `lib/` and `src/`
- **PHP Version**: 8.0+
- **Dependencies**: GuzzleHttp 6.2+ or 7.0+
- **Generator**: OpenAPI Generator v7.13.0

## Git Workflow

The repository uses a custom git push script:

```bash
./git_push.sh [user_id] [repo_id] "commit message" [git_host]
```

Default values are configured for the talisman/unleashed repository.
