<?php
/**
 * HeaderSelector
 * PHP version 8.1
 *
 * @category Class
 * @package  Temu
 * @author   OpenAPI Generator team
 * @link     https://openapi-generator.tech
 */

/**
 * Temu Open APIs
 *
 * Comprehensive API collection for Temu marketplace operations including product management, order processing, inventory management, pricing, and fulfillment services.  All endpoints use POST method with operation type specified in request body. Authentication uses MD5 signature with app_key, app_secret, access_token, and timestamp.
 *
 * The version of the OpenAPI document: 1.0.0
 * Generated by: https://openapi-generator.tech
 * OpenAPI Generator version: 7.1.0
 */

/**
 * NOTE: This class is auto generated by OpenAPI Generator (https://openapi-generator.tech).
 * https://openapi-generator.tech
 * Do not edit the class manually.
 */

namespace Temu;

/**
 * HeaderSelector Class Doc Comment
 *
 * Handles HTTP header selection and content negotiation for API requests.
 *
 * This class is responsible for:
 * - Selecting appropriate Accept headers based on available content types
 * - Setting Content-Type headers for requests
 * - Handling multipart/form-data requests (file uploads)
 * - Quality value (q-value) calculation for content negotiation
 * - Prioritizing JSON content types when available
 *
 * The HeaderSelector implements RFC 9110 (HTTP Semantics) content negotiation,
 * automatically assigning quality values to give preference to JSON responses
 * while supporting other content types as needed.
 *
 * @category Class
 * @package  Temu
 * @author   OpenAPI Generator team
 * @link     https://openapi-generator.tech
 * @link     https://www.rfc-editor.org/rfc/rfc9110.html#name-accept RFC 9110 Accept Header
 *
 * @example Basic Usage
 * ```php
 * $selector = new HeaderSelector();
 * $headers = $selector->selectHeaders(
 *     ['application/json', 'text/plain'],
 *     'application/json',
 *     false
 * );
 * // Returns: ['Accept' => 'application/json', 'Content-Type' => 'application/json']
 * ```
 *
 * @example File Upload (Multipart)
 * ```php
 * $headers = $selector->selectHeaders(
 *     ['application/json'],
 *     'multipart/form-data',
 *     true  // isMultipart
 * );
 * // Returns: ['Accept' => 'application/json']
 * // Note: Content-Type is omitted for multipart, added by HTTP client
 * ```
 */
class HeaderSelector
{
    /**
     * Select HTTP headers for an API request
     *
     * Generates appropriate HTTP headers based on acceptable content types and
     * request content type. Handles both standard JSON requests and multipart
     * file uploads.
     *
     * For the Accept header:
     * - Filters out empty values
     * - Returns single value if only one Accept type
     * - Prioritizes JSON types with quality values (q-values)
     * - Implements proper content negotiation per RFC 9110
     *
     * For the Content-Type header:
     * - Sets to specified value for standard requests
     * - Defaults to 'application/json' if not specified
     * - Omits for multipart requests (boundary set by HTTP client)
     *
     * @param string[] $accept      Array of acceptable content types (e.g., ['application/json', 'text/xml'])
     * @param string   $contentType Content-Type for the request body (e.g., 'application/json')
     * @param bool     $isMultipart Whether this is a multipart/form-data request (file upload)
     *
     * @return string[] Associative array of HTTP headers ['Header-Name' => 'value']
     *
     * @example Standard JSON request
     * ```php
     * $headers = $selector->selectHeaders(
     *     ['application/json'],
     *     'application/json',
     *     false
     * );
     * // ['Accept' => 'application/json', 'Content-Type' => 'application/json']
     * ```
     *
     * @example Multiple Accept types with quality values
     * ```php
     * $headers = $selector->selectHeaders(
     *     ['application/json', 'application/xml', 'text/plain'],
     *     'application/json',
     *     false
     * );
     * // ['Accept' => 'application/json,application/xml;q=0.9,text/plain;q=0.8',
     * //  'Content-Type' => 'application/json']
     * ```
     */
    public function selectHeaders(array $accept, string $contentType, bool $isMultipart): array
    {
        $headers = [];

        $acceptHeader = $this->selectAcceptHeader($accept);
        if ($acceptHeader !== null) {
            $headers['Accept'] = $acceptHeader;
        }

        if (!$isMultipart) {
            if($contentType === '') {
                $contentType = 'application/json';
            }

            $headers['Content-Type'] = $contentType;
        }

        return $headers;
    }

    /**
     * Return the header 'Accept' based on an array of Accept provided.
     *
     * @param string[] $accept Array of header
     *
     * @return null|string Accept (e.g. application/json)
     */
    private function selectAcceptHeader(array $accept): ?string
    {
        # filter out empty entries
        $accept = array_filter($accept);

        if (count($accept) === 0) {
            return null;
        }

        # If there's only one Accept header, just use it
        if (count($accept) === 1) {
            return reset($accept);
        }

        # If none of the available Accept headers is of type "json", then just use all them
        $headersWithJson = preg_grep('~(?i)^(application/json|[^;/ \t]+/[^;/ \t]+[+]json)[ \t]*(;.*)?$~', $accept);
        if (count($headersWithJson) === 0) {
            return implode(',', $accept);
        }

        # If we got here, then we need add quality values (weight), as described in IETF RFC 9110, Items 12.4.2/12.5.1,
        # to give the highest priority to json-like headers - recalculating the existing ones, if needed
        return $this->getAcceptHeaderWithAdjustedWeight($accept, $headersWithJson);
    }

    /**
    * Create an Accept header string from the given "Accept" headers array, recalculating all weights
    *
    * @param string[] $accept            Array of Accept Headers
    * @param string[] $headersWithJson   Array of Accept Headers of type "json"
    *
    * @return string "Accept" Header (e.g. "application/json, text/html; q=0.9")
    */
    private function getAcceptHeaderWithAdjustedWeight(array $accept, array $headersWithJson): string
    {
        $processedHeaders = [
            'withApplicationJson' => [],
            'withJson' => [],
            'withoutJson' => [],
        ];

        foreach ($accept as $header) {

            $headerData = $this->getHeaderAndWeight($header);

            if (stripos($headerData['header'], 'application/json') === 0) {
                $processedHeaders['withApplicationJson'][] = $headerData;
            } elseif (in_array($header, $headersWithJson, true)) {
                $processedHeaders['withJson'][] = $headerData;
            } else {
                $processedHeaders['withoutJson'][] = $headerData;
            }
        }

        $acceptHeaders = [];
        $currentWeight = 1000;

        $hasMoreThan28Headers = count($accept) > 28;

        foreach($processedHeaders as $headers) {
            if (count($headers) > 0) {
                $acceptHeaders[] = $this->adjustWeight($headers, $currentWeight, $hasMoreThan28Headers);
            }
        }

        $acceptHeaders = array_merge(...$acceptHeaders);

        return implode(',', $acceptHeaders);
    }

    /**
     * Given an Accept header, returns an associative array splitting the header and its weight
     *
     * @param string $header "Accept" Header
     *
     * @return array with the header and its weight
     */
    private function getHeaderAndWeight(string $header): array
    {
        # matches headers with weight, splitting the header and the weight in $outputArray
        if (preg_match('/(.*);\s*q=(1(?:\.0+)?|0\.\d+)$/', $header, $outputArray) === 1) {
            $headerData = [
                'header' => $outputArray[1],
                'weight' => (int)($outputArray[2] * 1000),
            ];
        } else {
            $headerData = [
                'header' => trim($header),
                'weight' => 1000,
            ];
        }

        return $headerData;
    }

    /**
     * @param array[] $headers
     * @param int     $currentWeight
     * @param bool    $hasMoreThan28Headers
     * @return string[] array of adjusted "Accept" headers
     */
    private function adjustWeight(array $headers, int &$currentWeight, bool $hasMoreThan28Headers): array
    {
        usort($headers, function (array $a, array $b) {
            return $b['weight'] - $a['weight'];
        });

        $acceptHeaders = [];
        foreach ($headers as $index => $header) {
            if($index > 0 && $headers[$index - 1]['weight'] > $header['weight'])
            {
                $currentWeight = $this->getNextWeight($currentWeight, $hasMoreThan28Headers);
            }

            $weight = $currentWeight;

            $acceptHeaders[] = $this->buildAcceptHeader($header['header'], $weight);
        }

        $currentWeight = $this->getNextWeight($currentWeight, $hasMoreThan28Headers);

        return $acceptHeaders;
    }

    /**
     * @param string $header
     * @param int    $weight
     * @return string
     */
    private function buildAcceptHeader(string $header, int $weight): string
    {
        if($weight === 1000) {
            return $header;
        }

        return trim($header, '; ') . ';q=' . rtrim(sprintf('%0.3f', $weight / 1000), '0');
    }

    /**
     * Calculate the next weight, based on the current one.
     *
     * If there are less than 28 "Accept" headers, the weights will be decreased by 1 on its highest significant digit, using the
     * following formula:
     *
     *    next weight = current weight - 10 ^ (floor(log(current weight - 1)))
     *
     *    ( current weight minus ( 10 raised to the power of ( floor of (log to the base 10 of ( current weight minus 1 ) ) ) ) )
     *
     * Starting from 1000, this generates the following series:
     *
     * 1000, 900, 800, 700, 600, 500, 400, 300, 200, 100, 90, 80, 70, 60, 50, 40, 30, 20, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1
     *
     * The resulting quality codes are closer to the average "normal" usage of them (like "q=0.9", "q=0.8" and so on), but it only works
     * if there is a maximum of 28 "Accept" headers. If we have more than that (which is extremely unlikely), then we fall back to a 1-by-1
     * decrement rule, which will result in quality codes like "q=0.999", "q=0.998" etc.
     *
     * @param int  $currentWeight varying from 1 to 1000 (will be divided by 1000 to build the quality value)
     * @param bool $hasMoreThan28Headers
     * @return int
     */
    public function getNextWeight(int $currentWeight, bool $hasMoreThan28Headers): int
    {
        if ($currentWeight <= 1) {
            return 1;
        }

        if ($hasMoreThan28Headers) {
            return $currentWeight - 1;
        }

        return $currentWeight - 10 ** floor( log10($currentWeight - 1) );
    }
}
