<?php
/**
 * AuthorizationApi
 * PHP version 8.1
 *
 * @category Class
 * @package  Temu
 * @author   Temu SDK Generator
 */

namespace Temu\Api;

use Temu\ApiException;
use Temu\Model\AccessTokenInfoRequest;
use Temu\Model\AccessTokenInfoResponse;
use Temu\Model\CreateAccessTokenRequest;
use Temu\Model\CreateAccessTokenResponse;

/**
 * AuthorizationApi Class
 *
 * Handles OAuth 2.0 authentication and token management for Temu API access.
 *
 * This API class provides methods for:
 * - Creating OAuth access tokens using client credentials
 * - Retrieving information about existing access tokens
 * - Managing token lifecycle and expiration
 *
 * All Temu API operations require a valid access token obtained through this API.
 * Tokens should be cached and reused until expiration to avoid unnecessary API calls.
 *
 * @category Class
 * @package  Temu
 * @author   Temu SDK Generator
 *
 * @example Basic Authentication Flow
 * ```php
 * use Temu\Api\AuthorizationApi;
 * use Temu\Model\CreateAccessTokenRequest;
 *
 * $authApi = new AuthorizationApi();
 *
 * // Step 1: Create access token
 * $request = new CreateAccessTokenRequest([
 *     'client_id' => 'your_client_id',
 *     'client_secret' => 'your_client_secret',
 *     'grant_type' => 'client_credentials'
 * ]);
 *
 * $response = $authApi->createAccessToken($request);
 * if ($response->getSuccess()) {
 *     $token = $response->getResult()->getAccessToken();
 *     $expiresIn = $response->getResult()->getExpiresIn();
 *
 *     // Step 2: Store token for future use
 *     $_SESSION['temu_token'] = $token;
 *     $_SESSION['temu_token_expires'] = time() + $expiresIn;
 * }
 * ```
 */
class AuthorizationApi extends DefaultApi
{
    /**
     * Get access token information
     *
     * Retrieves detailed information about the current access token including
     * expiration time, scope, and validity status. Useful for checking if a
     * token needs to be refreshed before making API calls.
     *
     * **Operation Type:** `bg.oauth.token.info.get`
     *
     * @param AccessTokenInfoRequest $request Request object for access token info
     *
     * @return AccessTokenInfoResponse Response containing token information
     *                                 - access_token: The token string
     *                                 - expires_at: Unix timestamp when token expires
     *                                 - scope: Token permissions/scope
     *
     * @throws ApiException If HTTP request fails or network error occurs
     *
     * @example Check token expiration
     * ```php
     * $request = new AccessTokenInfoRequest();
     * $response = $authApi->getAccessTokenInfo($request);
     *
     * if ($response->getSuccess()) {
     *     $tokenInfo = $response->getResult();
     *     $expiresAt = $tokenInfo->getExpiresAt();
     *
     *     if ($expiresAt < time() + 300) { // Expires in less than 5 minutes
     *         echo "Token needs refresh";
     *     }
     * }
     * ```
     */
    public function getAccessTokenInfo(AccessTokenInfoRequest $request): AccessTokenInfoResponse
    {
        /** @var AccessTokenInfoResponse $response */
        $response = $this->rootPost($request, 'application/json', $request->getResponseClass());
        return $response;
    }

    /**
     * Create access token
     *
     * Creates a new OAuth 2.0 access token using client credentials grant flow.
     * This is the primary method for authenticating with the Temu API.
     *
     * **Operation Type:** `bg.oauth.token.create`
     *
     * **Important Notes:**
     * - Tokens have a limited lifetime (typically 2 hours)
     * - Cache tokens and reuse them until expiration
     * - Implement token refresh logic in production applications
     * - Never expose client_secret in client-side code
     *
     * @param CreateAccessTokenRequest $request Request object with client credentials
     *                                         Required fields:
     *                                         - client_id: Your application client ID
     *                                         - client_secret: Your application secret
     *                                         - grant_type: 'client_credentials'
     *
     * @return CreateAccessTokenResponse Response containing the access token
     *                                   - access_token: Use this for API authentication
     *                                   - expires_in: Seconds until token expires
     *                                   - token_type: Usually "Bearer"
     *
     * @throws ApiException If authentication fails or network error occurs
     *
     * @example Create and store access token
     * ```php
     * $request = new CreateAccessTokenRequest([
     *     'client_id' => getenv('TEMU_CLIENT_ID'),
     *     'client_secret' => getenv('TEMU_CLIENT_SECRET'),
     *     'grant_type' => 'client_credentials'
     * ]);
     *
     * try {
     *     $response = $authApi->createAccessToken($request);
     *
     *     if ($response->getSuccess()) {
     *         $accessToken = $response->getResult()->getAccessToken();
     *         $expiresIn = $response->getResult()->getExpiresIn();
     *
     *         // Configure SDK with new token
     *         $config = Configuration::getDefaultConfiguration();
     *         $config->setAccessToken($accessToken);
     *
     *         // Store token with expiration
     *         file_put_contents('token.json', json_encode([
     *             'token' => $accessToken,
     *             'expires_at' => time() + $expiresIn
     *         ]));
     *     } else {
     *         echo "Auth failed: " . $response->getErrorMsg();
     *     }
     * } catch (ApiException $e) {
     *     echo "HTTP Error: " . $e->getMessage();
     * }
     * ```
     *
     * @example Token refresh pattern
     * ```php
     * class TokenManager {
     *     private $authApi;
     *     private $tokenFile = 'token.json';
     *
     *     public function getValidToken(): string {
     *         $data = json_decode(file_get_contents($this->tokenFile), true);
     *
     *         // Refresh if expired or expiring soon
     *         if (time() >= $data['expires_at'] - 300) {
     *             $this->refreshToken();
     *             $data = json_decode(file_get_contents($this->tokenFile), true);
     *         }
     *
     *         return $data['token'];
     *     }
     *
     *     private function refreshToken() {
     *         $request = new CreateAccessTokenRequest([...]);
     *         $response = $this->authApi->createAccessToken($request);
     *         // Save new token...
     *     }
     * }
     * ```
     */
    public function createAccessToken(CreateAccessTokenRequest $request): CreateAccessTokenResponse
    {
        /** @var CreateAccessTokenResponse $response */
        $response = $this->rootPost($request, 'application/json', $request->getResponseClass());
        return $response;
    }
}
