# Configuration Guide

Complete guide for configuring the Temu PHP SDK.

## Table of Contents

- [Quick Start](#quick-start)
- [Regional Endpoints](#regional-endpoints)
- [Authentication](#authentication)
- [Environment Selection](#environment-selection)
- [Advanced Configuration](#advanced-configuration)
- [Constants Reference](#constants-reference)

## Quick Start

```php
use Temu\Configuration;
use Temu\Api\OrderManagementApi;
use Temu\Model\OrderListRequest;

// Create configuration for US production
$config = new Configuration();
$config->setAccessToken('your-access-token-here');

// Use with API client
$orderApi = new OrderManagementApi(null, $config);

// Create request object
$request = new OrderListRequest([
    'page_num' => 1,
    'page_size' => 50
]);

// Make API call
$response = $orderApi->getOrderList($request);
if ($response->getSuccess()) {
    $orders = $response->getResult()->getOrderList();
}
```

## Regional Endpoints

The SDK supports multiple regional endpoints for optimal performance.

### Available Regions

| Region | Constant | Code | Endpoint |
|--------|----------|------|----------|
| United States / North America | `Configuration::REGION_US` | `'us'` | `https://openapi-b-us.temu.com/openapi/router` |
| Europe | `Configuration::REGION_EU` | `'eu'` | `https://openapi-b-eu.temu.com/openapi/router` |
| Asia Pacific | `Configuration::REGION_APAC` | `'ap'` | `https://openapi-b-ap.temu.com/openapi/router` |
| US Sandbox | `Configuration::HOST_US_SANDBOX` | - | `https://openapi-sandbox-us.temu.com/openapi/router` |

### Setting Region by Code

```php
$config = new Configuration();

// US Production (default)
$config->setHostByRegion(Configuration::REGION_US);

// EU Production
$config->setHostByRegion(Configuration::REGION_EU);

// APAC Production
$config->setHostByRegion(Configuration::REGION_APAC);
```

### Setting Region by Index

```php
$config = new Configuration();

// Using constants (recommended)
$config->setHostByIndex(Configuration::HOST_EU_PRODUCTION);
$config->setHostByIndex(Configuration::HOST_APAC_PRODUCTION);

// Using numeric index
$config->setHostByIndex(0); // US Production
$config->setHostByIndex(1); // EU Production
$config->setHostByIndex(2); // APAC Production
$config->setHostByIndex(3); // US Sandbox
```

## Authentication

### Access Token

All Temu API requests require an access token obtained through OAuth 2.0.

```php
use Temu\Configuration;
use Temu\Api\AuthorizationApi;
use Temu\Model\CreateAccessTokenRequest;

// Step 1: Create access token using client credentials
$authApi = new AuthorizationApi();

$request = new CreateAccessTokenRequest([
    'client_id' => 'your-client-id',
    'client_secret' => 'your-client-secret',
    'grant_type' => 'client_credentials'
]);

$response = $authApi->createAccessToken($request);

if ($response->getSuccess()) {
    $accessToken = $response->getResult()->getAccessToken();
    $expiresIn = $response->getResult()->getExpiresIn();

    // Step 2: Configure client with access token
    $config = new Configuration();
    $config->setAccessToken($accessToken);

    // Store token for reuse (implement token refresh logic)
    // Token expires after $expiresIn seconds
} else {
    echo "Auth failed: " . $response->getErrorMsg();
}
```

### API Keys (if applicable)

```php
$config->setApiKey('api_key_identifier', 'your-api-key');
$config->setApiKeyPrefix('api_key_identifier', 'Bearer');
```

## Environment Selection

### Production Environment

```php
// US Production (default)
$config = new Configuration();
$config->setHostByRegion(Configuration::REGION_US);

// EU Production
$config->setHostByRegion(Configuration::REGION_EU);

// APAC Production
$config->setHostByRegion(Configuration::REGION_APAC);
```

### Sandbox/Testing Environment

```php
// US Sandbox for testing
$config = new Configuration();
$config->setHostByRegion(Configuration::REGION_US, true);
$config->setAccessToken('your-sandbox-token');
```

## Advanced Configuration

### Debug Mode

Enable debug mode to log HTTP requests and responses:

```php
$config = new Configuration();
$config->setDebug(true);
$config->setDebugFile('/path/to/debug.log');
```

### User Agent

Customize the HTTP User-Agent header:

```php
$config->setUserAgent('MyApp/1.0 (Temu SDK)');
```

### Temporary Files

Set custom temporary folder for file operations:

```php
$config->setTempFolderPath('/custom/temp/path');
```

### Custom Host

Manually set a custom host URL:

```php
$config->setHost('https://custom-endpoint.example.com/api');
```

## Constants Reference

### Host/Endpoint Indices

```php
Configuration::HOST_US_PRODUCTION    // 0 - US Production
Configuration::HOST_EU_PRODUCTION    // 1 - EU Production
Configuration::HOST_APAC_PRODUCTION  // 2 - APAC Production
Configuration::HOST_US_SANDBOX       // 3 - US Sandbox
```

### Region Codes

```php
Configuration::REGION_US    // 'us' - United States / North America
Configuration::REGION_EU    // 'eu' - Europe
Configuration::REGION_APAC  // 'ap' - Asia Pacific
```

### Boolean Formats

```php
Configuration::BOOLEAN_FORMAT_INT     // 'int' - Use 0/1 for booleans
Configuration::BOOLEAN_FORMAT_STRING  // 'string' - Use 'true'/'false' for booleans
```

## Complete Example

```php
use Temu\Configuration;
use Temu\Api\OrderManagementApi;
use Temu\Api\ProductsManagementApi;
use Temu\Model\OrderListRequest;
use Temu\Model\GoodsListRequest;
use Temu\ApiException;

// Configure for EU production with debug logging
$config = new Configuration();
$config->setHostByRegion(Configuration::REGION_EU);
$config->setAccessToken('your-access-token');
$config->setDebug(true);
$config->setDebugFile(__DIR__ . '/logs/api-debug.log');
$config->setUserAgent('MyStore/2.0 (Temu SDK)');

// Use with multiple API clients
$orderApi = new OrderManagementApi(null, $config);
$productApi = new ProductsManagementApi(null, $config);

// Make API calls
try {
    // Get orders
    $orderRequest = new OrderListRequest([
        'page_num' => 1,
        'page_size' => 50
    ]);
    $orderResponse = $orderApi->getOrderList($orderRequest);

    if ($orderResponse->getSuccess()) {
        $orders = $orderResponse->getResult()->getOrderList();
        echo "Orders retrieved: " . count($orders) . "\n";
    }

    // Get products
    $productRequest = new GoodsListRequest([
        'page_num' => 1,
        'page_size' => 50
    ]);
    $productResponse = $productApi->getGoodsList($productRequest);

    if ($productResponse->getSuccess()) {
        $products = $productResponse->getResult()->getGoodsList();
        echo "Products retrieved: " . count($products) . "\n";
    }

} catch (ApiException $e) {
    echo "HTTP Error: " . $e->getMessage() . "\n";
    echo "Response: " . $e->getResponseBody() . "\n";
}
```

## Default Configuration

You can also set a default configuration instance:

```php
use Temu\Configuration;

// Set default configuration
$defaultConfig = new Configuration();
$defaultConfig->setHostByRegion(Configuration::REGION_EU);
$defaultConfig->setAccessToken('your-token');
Configuration::setDefaultConfiguration($defaultConfig);

// Get default configuration
$config = Configuration::getDefaultConfiguration();

// All API instances will use this configuration by default
$orderApi = new OrderManagementApi(); // Uses default config
```

## Troubleshooting

### Check Current Configuration

```php
// Get debug report
echo Configuration::toDebugReport();

// Output:
// PHP SDK (Temu) Debug Report:
//     OS: Darwin 24.6.0 ...
//     PHP Version: 8.1.0
//     The version of the OpenAPI document: 1.0.0
//     SDK Package Version: 1.0
//     Temp Folder Path: /tmp
```

### List Available Endpoints

```php
$config = new Configuration();
$hosts = $config->getHostSettings();

foreach ($hosts as $index => $host) {
    echo "[$index] {$host['description']}\n";
    echo "      {$host['url']}\n\n";
}
```

## See Also

### SDK Documentation
- [Developer Guide](../DEVELOPER_GUIDE.md) - Comprehensive guide with architecture, best practices, and troubleshooting
- [API Documentation](../API_DOCUMENTATION.md) - Complete API reference for all 11 API classes
- [SDK Summary](../SDK_SUMMARY.md) - SDK implementation overview
- [Usage Examples](../USAGE_EXAMPLE.php) - Working code examples for all API operations

### External Resources
- [Temu Partner Documentation](https://partner.temu.com/documentation) - Official Temu API documentation
