# CLAUDE.md

This file provides guidance to Claude Code (claude.ai/code) when working with code in this repository.

## Project Overview

This repository contains a PHP SDK for Temu Open APIs, generated from an OpenAPI v3 specification. The library provides comprehensive access to Temu marketplace operations including product management, order processing, inventory management, pricing, and fulfillment services.

## Build Commands

### Regenerate PHP Library
```bash
./openapi/build.sh
```
This regenerates the entire PHP library from the OpenAPI specification using OpenAPI Generator v7.1.0.

### Testing
```bash
composer install
vendor/bin/phpunit                    # Run all tests
vendor/bin/phpunit --filter ClassName # Run specific test class
```

### Code Style
```bash
vendor/bin/php-cs-fixer fix          # Fix code style issues
vendor/bin/php-cs-fixer fix --dry-run # Check style issues without fixing
```

## Architecture Overview

### Unique API Design Pattern
Temu APIs use a single POST endpoint pattern where:
- **Base URL**: Region-specific endpoints (US, EU, APAC, Sandbox)
  - US Production: `https://openapi-b-us.temu.com/openapi/router`
  - EU Production: `https://openapi-b-eu.temu.com/openapi/router`
  - APAC Production: `https://openapi-b-ap.temu.com/openapi/router`
  - US Sandbox: `https://openapi-sandbox-us.temu.com/openapi/router`
- **Method**: All operations use POST
- **Operation Selection**: Specified via `type` field in request body (e.g., `"bg.order.list.get"`, `"bg.local.goods.list.get"`)
- **Authentication**: OAuth 2.0 Bearer token authentication

### Library Architecture (Refactored)
The SDK uses an efficient inheritance pattern with **60% code reduction**:

- **`lib/Api/DefaultApi.php`** - Base API client with `rootPost()` method
- **11 Specialized API Classes** (all extend DefaultApi):
  1. `AuthorizationApi` - OAuth token management (2 methods)
  2. `ProductsListingApi` - Product creation and listing (11 methods)
  3. `ProductsManagementApi` - Product information management (8 methods)
  4. `InventoryManagementApi` - Stock/inventory management (1 method)
  5. `PricingApi` - Product pricing and price orders (11 methods)
  6. `OrderManagementApi` - Order retrieval and details (5 methods)
  7. `OrderFulfillmentApi` - Shipment and fulfillment (3 methods)
  8. `RefundAndReturnApi` - Returns and refunds (3 methods)
  9. `BuyShippingApi` - Shipping label and warehouse management (7 methods)
  10. `WebhookApi` - Event subscription (1 method)
  11. `DefaultApi` - Base HTTP client

- **112 Model Classes** organized by category:
  - Authentication models (2 models)
  - Product models (44 models)
  - Order models (20 models)
  - Pricing models (14 models)
  - Shipping & Logistics models (18 models)
  - Return & Refund models (8 models)
  - Base models: `BaseRequest`, `BaseResponse`, `PostRequest`

- **Core Utilities**:
  - `lib/Configuration.php` - Regional endpoints, authentication, debug mode
  - `lib/ObjectSerializer.php` - JSON serialization with DateTime and file handling
  - `lib/ApiException.php` - HTTP error handling with response access
  - `lib/HeaderSelector.php` - Content negotiation (RFC 9110)

### OpenAPI Source Files
- **`openapi/openapi.yaml`** - OpenAPI v3 specification (converted from Postman collection)
- **`openapi/temu.collection`** - Original Postman collection (3500+ lines)
- **`openapi/config.json`** - Generator configuration
- **`openapi/build.sh`** - Build script for regeneration

### Operation Type Patterns
API operations follow domain-based naming:
- `bg.open.*` - Authentication operations
- `bg.local.goods.*` - Product management
- `bg.order.*` - Order operations
- `bg.logistics.*` - Shipping and fulfillment
- `bg.aftersales.*` - Returns and refunds

### Authentication Flow
OAuth 2.0 Bearer token authentication:
1. Use `AuthorizationApi::createAccessToken()` with client credentials
2. Receive access token with expiration time
3. Configure SDK: `Configuration::setAccessToken($token)`
4. Token is automatically included in all API requests as Bearer token
5. Implement token refresh logic before expiration

## Documentation

The SDK includes comprehensive documentation:

### Primary Documentation Files
- **`DEVELOPER_GUIDE.md`** - Complete developer guide (600+ lines)
  - Quick start, architecture, API reference, best practices, troubleshooting
- **`API_DOCUMENTATION.md`** - Detailed API reference for all 11 API classes
- **`SDK_SUMMARY.md`** - SDK implementation overview and architecture
- **`REFACTORING_SUMMARY.md`** - Code refactoring details and inheritance pattern
- **`USAGE_EXAMPLE.php`** - Working examples for all API classes

### Inline Documentation
All core classes include comprehensive PHPDoc comments:
- Core utilities: `Configuration`, `ApiException`, `ObjectSerializer`, `HeaderSelector`
- API classes: Detailed method documentation with examples and operation types
- Base models: `BaseRequest`, `BaseResponse`, `PostRequest` with usage patterns

### Generated Documentation
- **`docs/Api/`** - Auto-generated API class documentation
- **`docs/Model/`** - Auto-generated model class documentation

## Important Notes

### Code Generation
- Library is auto-generated - **do not manually edit** files in `lib/Api/`, `lib/Model/`, `docs/`, or `test/`
- Regeneration will overwrite generated files
- **Exception**: Enhanced PHPDoc comments in key files should be preserved
- Customizations should be made to OpenAPI spec or generator config

### Manual Documentation Files (Safe to Edit)
- `DEVELOPER_GUIDE.md`
- `API_DOCUMENTATION.md`
- `SDK_SUMMARY.md`
- `REFACTORING_SUMMARY.md`
- `USAGE_EXAMPLE.php`
- `CLAUDE.md` (this file)
- `CONFIGURATION.md`

### Package Details
- **Namespace**: `Temu\`
- **PHP Requirement**: ^8.1
- **Dependencies**: Guzzle HTTP client (^7.3), PSR-7 (^1.7 || ^2.0)
- **Package Name**: `talisman/temu`
- **Generated By**: OpenAPI Generator v7.1.0
- **Total Classes**: 11 API classes + 112 model classes + 4 core utilities