<?php

/*
 * Copyright Talisman Innovations Ltd. (2016). All rights reserved.
 */

namespace SkuVault;

use DateTime;
use DateTimeZone;

class Utils {

    protected $timezone;
    protected $apiUrl;
    protected $utils;
    

    public function __construct() {
        $this->apiUrl = 'https://app.skuvault.com/api/';
        $this->timezone = 'UTC';
    }
    
     /**
     * Get the Integrations (accounts)
     * @return json
     */
     
    public function getIntegrations() {

        $data = [
            'TenantToken' => $this->tenantToken,
            'UserToken' => $this->userToken
        ];

        return $this->remote->call($data, 'integration/getIntegrations');
    }
    
    
    /**
     * Get unit weights
     * used in product creation
     * @return array
     */
    public function getWeightUnits() {
        return ['lbs', 'kgs', 'oz', 'g'];
    }

    /**
     * Correct datetime format for SkuVault
     * @param datetime $dateTime
     * @return utc date 0001-01-01T00:00:00.0000000Z
     */
    public function formatDate($dateTime) {
        if (!$dateTime) {
            return;
        }
        
        $timezone = new DateTimeZone($this->timezone);
        $myDateTime = new DateTime($dateTime, $timezone);

        return $myDateTime->format('Y-m-d\TH:i:s.u\Z');
    }
    
    /**
     * Get available statuses
     * @return array
     */
    public function getSalesStatus() {
        return ['Pending', 'ReadyToShip', 'Completed', 'Cancelled', 'Invalid', 'ShippedUnpaid'];
    }

    /**
     * Get available checkout statues
     * @return array
     */
    public function getCheckoutStatuses() {
        return ['NotVisited', 'Visited', 'OnHold', 'Completed', 'CompletedOffline', 'Cancelled'];
    }

    /**
     * Get available payment statues
     * @return array
     */
    public function getPaymentStatus() {
        return ['NotSubmitted', 'Submitted', 'Deposited', 'Cleared', 'Failed'];
    }

    /**
     * Get available sale states
     * @return array
     */
    public function getSaleState() {
        return ['Active', 'Cancelled', 'Archived'];
    }

    public function getPoSendStaus() {
        return ['NotSent', 'Sent', 'NeedToResend'];
    }

    public function getPoPaymentStaus() {
        return ['NonePaid,', 'PartiallyPaid', 'FullyPaid'];
    }

    function getTimezone() {
        return $this->timezone;
    }

    function getApiUrl() {
        return $this->apiUrl;
    }

    public function convertToJson($object){     
        $class = get_class($object);
        $json = array();
        $methods = get_class_methods($class);
        $plength = count($methods);
        $json = array();
        for($i=0;$i<$plength;$i++){
            if(stripos($methods[$i], "get")!==FALSE){
                $property = mb_substr($methods[$i], 3,mb_strlen($methods[$i],'UTF-8'),'UTF-8');
                $setter = "set".mb_substr($methods[$i], 3,mb_strlen($methods[$i],'UTF-8'),'UTF-8');
                if(method_exists($object,$setter)){
                    $json[$property] = $object->$methods[$i]();
                }
                
            }
        }
        
        return $json;
    }
 
    
}
