<?php

/*
 * Copyright Talisman Innovations Ltd. (2016). All rights reserved.
 */

namespace SkuVault;

use SkuVault\Remote;
use SkuVault\Utils;
use SkuVault\Exception;
use DateTime;
use Psr\Log\LoggerInterface;

class Sales {

    protected $remote;
    protected $pathPart;
    protected $utils;
    protected $tenantToken;
    protected $userToken;

    /**
     * Setup
     * @param type $tenantToken
     * @param type $userToken
     */
    public function __construct($tenantToken, $userToken, LoggerInterface $logger) {
        $this->remote = new Remote($logger);
        $this->utils = new Utils();
        $this->pathPart = 'sales';
        $this->tenantToken = $tenantToken;
        $this->userToken = $userToken;
    }

    /**
     * Create Holds
     * 
     * $holds = [];
     * 
     * for($i=0;$i<10;$i++) {
     * $holds[] = $this->setHold($sku, $qty, $expireDate, $desc);
     * }
     * 
     * $this->createHolds($holds);
     * 
     * @param array $holds
     * @return json
     */
    public function createHolds(array $holds) {

        if (!$holds || empty($holds)) {
            throw new Exception('Holds array required');
        }

        $data = [
            'Holds' => $holds,
            'TenantToken' => $this->tenantToken,
            'UserToken' => $this->userToken,
        ];

        $result = $this->remote->call($data, $this->pathPart . '/createHolds');
        return $result;
    }

    public function setHold($sku, $qty, $expireDate, $desc) {
        return ['Sku' => $sku, 'Quantity' => $qty, 'ExpirationDateUtc' => $this->utils->formatDate($expireDate), 'Description' => $desc];
    }

    /**
     * Release Held Objects
     * 
     * $holds = [];
     * 
     * for($i=0;$i<10;$i++) {
     * $holds[] = $this->releaseHelds($sku, $qty);
     * }
     * 
     * $this->releaseHeldQuantities($holds)
     * 
     * @param array $skus
     * @return json
     */
    public function releaseHeldQuantities($skus) {

        if (!$skus || empty($skus)) {
            throw new Exception('Skus array required');
        }

        $data = [
            'SkusToRelease' => $skus,
            'TenantToken' => $this->tenantToken,
            'UserToken' => $this->userToken,
        ];

        $result = $this->remote->call($data, $this->pathPart . '/releaseHeldQuantities');
        return $result;
    }

    public function releaseHelds($sku, $qty) {
        return [$sku => $qty];
    }

    /**
     * Get Sales By Date
     * Note this call covers a 7 day period only
     * @param type $pageNumber
     * @param type $fromDate
     * @param type $toDate
     * @return Object
     * @throws Exception
     */        
    public function getSalesByDate($pageNumber, $fromDate = null, $toDate = null) {

        if ((!is_null($fromDate) && DateTime::createFromFormat('Y-m-d G:i:s', $fromDate) === FALSE) || (!is_null($toDate) && DateTime::createFromFormat('Y-m-d G:i:s', $toDate) === FALSE)) {
            throw new Exception('Check the entered date format');
        }

        $data = [];

        if (!is_null($fromDate)) {
            $data['FromDate'] = $this->utils->formatDate($fromDate);
        }
        if (!is_null($toDate)) {
            $data['ToDate'] = $this->utils->formatDate($toDate);
        }
        if (!is_null($pageNumber)) {
            $data['PageNumber'] = $pageNumber;
        }
        
        $data['TenantToken'] = $this->tenantToken;
        $data['UserToken'] = $this->userToken;

        $result = $this->remote->call($data, $this->pathPart . '/getSalesByDate');
        return $result;
    }
    
    /**
     * Depages Get Sales By Date
     * Note this call covers a 7 day period only
     * @param type $fromDate
     * @param type $toDate
     * @return Object
     */
    public function getAllSalesByDate($fromDate = null, $toDate = null) {

        $items = [];

        $pageNo = 0;
        while (1) {
            $data = $this->getSoldItems($pageNo, $fromDate, $toDate);

            $pageNo++;
            if (empty($data->SoldItems)) {
                break;
            }
            $items = array_merge($items, $data->SoldItems);
        }

        if (empty($items)) {
            $items = null;
        }

        $result = (object) ['SoldItems' => $items];

        return $result;
    }
    
    /**
     * De-weeked Get Sales By Date
     * this call will get all sales data between
     * gets around the 7 day limitation on the getSalesByDate call.
     * a start and end date.
     * @param type $fromDate
     * @param type $toDate
     * @return object
     */
    public function getAllSalesBetweenDates($fromDate, $toDate) {
            
        $from = new DateTime($fromDate);
        $end = $toDate;
        $array = [];
        
        # Split date range into 7 day periods
        while(1) {
            if(strtotime($from->format('Y-m-d H:i:s')) > strtotime($end)) {
                break;
            }
            $dateArray = [];
            $dateArray ['from'] = $from->format('Y-m-d H:i:s');
            $dateArray ['to'] = $from->modify('+1 week')->format('Y-m-d 23:59:59');
            $array[] = $dateArray;
            $from = $from->modify('next day'); 
        }
        
        $salesArray = [];
        
        foreach($array as $dates) {
     
            $sales = $this->getAllSalesByDate($dates['from'], $dates['to']);
            
            if(is_null($sales) || empty($sales->SoldItems)) {
                continue;
            }
            
            $salesArray = array_merge($salesArray, $sales->SoldItems);
            
        }
        
        return (object)['SoldItems'=>$salesArray];
    }
    

    /**
     * Get sales by status
     * Status is not a required field
     * see getSalesStatus() for available statuses
     * @param string $status
     * @return json
     */
    public function getSalesByStatus($status = null) {

        $data = [];

        if (!is_null($status)) {
            $data['Status'] = $status;
        }

        $data['TenantToken'] = $this->tenantToken;
        $data['UserToken'] = $this->userToken;

        $result = $this->remote->call($data, $this->pathPart . '/getSales');
        return $result;
    }

    /**
     * Get Sale Status
     * This call returns a list of sales and their statuses. 
     * @param array $orderIds ['id1','id2']
     * @return json
     * @throws Exception
     */
    public function getOnlineSaleStatus($orderIds) {

        if (!$orderIds || empty($orderIds)) {
            throw new Exception('Order Id array is required');
        }

        $data = [
            'OrderIds' => $orderIds,
            'TenantToken' => $this->tenantToken,
            'UserToken' => $this->userToken,
        ];

        $result = $this->remote->call($data, $this->pathPart . '/getOnlineSaleStatus');
        return $result;
    }

    /**
     * 
     * @param datetime $startDateUtc
     * @param datetime $endDateUtc
     * @param bool $breakDownKits
     * @return json
     * @throws Exception
     */
    public function getSoldItems($pageNumber = null, $startDateUtc, $endDateUtc, $breakDownKits = null) {

        if ((!is_null($startDateUtc) && DateTime::createFromFormat('Y-m-d G:i:s', $startDateUtc) === FALSE) || (!is_null($endDateUtc) && DateTime::createFromFormat('Y-m-d G:i:s', $endDateUtc) === FALSE)) {
            throw new Exception('Check the entered date format');
        }

        $data = [];

        $data['StartDateUtc'] = $this->utils->formatDate($startDateUtc);
        $data['EndDateUtc'] = $this->utils->formatDate($endDateUtc);
        if ($breakDownKits) {
            $data['BreakDownKits'] = $breakDownKits;
        }
        if($pageNumber) {
            $data['PageNumber'] = $pageNumber;
        }
        $data['TenantToken'] = $this->tenantToken;
        $data['UserToken'] = $this->userToken;

        $result = $this->remote->call($data, $this->pathPart . '/getSoldItems');
        return $result;
    }
    
    
    /**
     * Get All Sold Items depaged
     * @param int $pageNumber
     * @return object
     */
    public function getAllSoldItems($startDateUtc, $endDateUtc, $breakDownKits = null) {

       $items = [];

        $pageNo = 0;
        while (1) {
            $data = $this->getSoldItems($pageNo, $startDateUtc, $endDateUtc, $breakDownKits);
            $pageNo++;
            if (empty($data->SoldItems)) {
                break;
            }
            $items = array_merge($items, $data->SoldItems);
        }
        
        if(empty($items)) {
            $items = null;
        }
        
        $result = (object)['SoldItems'=>$items];

        return $result;
    }
    
    
    /**
     * Get checkout status
     * @param type $status
     * @return string
     */
    protected function checkoutStatus($status) {
        
        # Potenial Statuses
        # NotVisited, Visited, OnHold, Completed, CompletedOffline, Cancelled
        
        switch ($status) {
            case 'CLOSED':
                $status = 'Completed';
                break;
            case 'VOIDED':
                $status = 'Cancelled';
                break;
            case 'SAVED':
                $status = 'OnHold';
                break;
            default :
                $status = 'Completed';
                break;
        }

        return $status;
    }
    

    /**
     * Sync Sale
     * inout example
     * create items itemsSkus  array fullfilled items use the same code
     * $itemSkus = [];
     * foreach($products as $product) {
     * $itemSkus[] = $this->createItems($sku, $qty, $price);
     * }
     * $sale = $this->createSale($orderId, $orderDateUTC, $OrderTotal, $checkoutStatus, $paymentStatus, $saleState, $itemSkus, $fulfilledItems, $warehouseId);
     * $this->syncOnlineSale($sale);
     * @return json
     * @throws Exception
     */
    public function syncOnlineSale($sale) {

        if (!$sale || empty($sale)) {
            throw new Exception('Sale data required');
        }
        $sale['TenantToken'] = $this->tenantToken;
        $sale['UserToken'] = $this->userToken;

        $result = $this->remote->call($sale, $this->pathPart . '/syncOnlineSale');
        return $result;
    }

    public function createSale($orderId, $orderDateUTC = null, $OrderTotal = null, $checkoutStatus = null, $paymentStatus = null, $saleState = null, $itemSkus, $fulfilledItems = null, $warehouseId = null, $shippingInfo = null) {

        if (!$orderId || !$itemSkus || empty($itemSkus)) {
            throw new Exception('Sale data required');
        }

        $sale = [];
        $sale['OrderId'] = $orderId;
        if ($orderDateUTC) {
            $sale['OrderDateUtc'] = $this->utils->formatDate($orderDateUTC);
        }
        if ($OrderTotal) {
            $sale['OrderTotal'] = (float) number_format($OrderTotal, 2, '.', '');
        }
        if ($checkoutStatus) {
            $sale['CheckoutStatus'] = $checkoutStatus;
        }
        if ($paymentStatus) {
            $sale['PaymentStatus'] = $paymentStatus;
        }
        if ($saleState) {
            $sale['SaleState'] =  $saleState;
        }
        if ($fulfilledItems) {
            $sale['FulfilledItems'] = $fulfilledItems;
        }
        if ($warehouseId) {
            $sale['WarehouseId'] = $warehouseId;
        }
        
        if($shippingInfo) {
            $sale['ShippingInfo'] = $shippingInfo;
        }
        
        $sale['ItemSkus'] = $itemSkus;

        return $sale;
    }

    public function createItems($sku, $qty, $price) {

        if (!$sku || !$qty) {
            throw new Exception('Create Product data missing');
        }
        $addProduct = ['Sku' => $sku, 'Quantity' => (int) $qty, 'UnitPrice' => (float) number_format($price, 2, '.', '')];
        return $addProduct;
    }
    
    public function createShippingInfo($ShippingStatus = null, $ShippingCarrier = null, $ShippingClass = null, $FirstName = null, $LastName = null, $CompanyName = null, $PhoneNumber = null, $Email = null, $Line1 = null, $Line2 = null, $City = null, $Region = null, $Postal = null, $Country = null) {

        $shipping = [];
        if($ShippingStatus) { $shipping['ShippingStatus'] = $ShippingStatus; }
        if($ShippingCarrier) { $shipping['ShippingCarrier'] = $ShippingCarrier; }
        if($ShippingClass) { $shipping['ShippingClass'] = $ShippingClass; }
        if($FirstName) { $shipping['FirstName'] = $FirstName; }
        if($LastName) { $shipping['LastName'] = $LastName; }
        if($CompanyName) { $shipping['CompanyName'] = $CompanyName; }
        if($PhoneNumber) { $shipping['PhoneNumber'] = $PhoneNumber; }
        if($Email) { $shipping['Email'] = $Email; }
        if($Line1) { $shipping['Line1'] = $Line1; }
        if($Line2) { $shipping['Line2'] = $Line2; }
        if($City) { $shipping['City'] = $City; }
        if($Region) { $shipping['Region'] = $Region; }
        if($Postal) { $shipping['Postal'] = $Postal; }
        if($Country) { $shipping['Country'] = $Country; }
        
        return $shipping;
    }

    /**
     * Sync Online sales
     * create items itemsSkus  array fullfilled items use the same code
     * $itemSkus = [];
     * foreach($products as $product) {
     * $itemSkus[] = $this->createItems($sku, $qty, $price);
     * }
     * $sales = []
     * $sales[] = $this->createSale($orderId, $orderDateUTC, $OrderTotal, $checkoutStatus, $paymentStatus, $saleState, $itemSkus, $fulfilledItems, $warehouseId);
     * $this->syncOnlineSale($sales);
     * @return json
     * @throws Exception
     */
    public function syncOnlineSales($sales) {

        if (!$sale || empty($sales)) {
            throw new Exception('Sales array required');
        }

        $data = [
            'Sales' => $sales,
            'TenantToken' => $this->tenantToken,
            'UserToken' => $this->userToken,
        ];

        $result = $this->remote->call($data, $this->pathPart . '/syncOnlineSales');
        return $result;
    }

    /**
     * Sync sale and remove items 
     * @param \SkuVault\Objects\Sale $sale
     * @return json
     * @throws Exception
     */
    public function syncShippedSaleAndRemoveItems($sale) {

        if (!$sale || empty($sale)) {
            throw new Exception('Sale data required');
        }

        $sale['TenantToken'] = $this->tenantToken;
        $sale['UserToken'] = $this->userToken;

        $result = $this->remote->call($sale, $this->pathPart . '/syncShippedSaleAndRemoveItems');
        return $result;
    }

    /**
     * Update Sales Status
     * options are Pending, ReadyToShip, Completed, Cancelled, Invalid, ShippedUnpaid
     * @param string $saleId
     * @param string $status
     * @return json
     * @throws Exception
     */
    public function updateOnlineSaleStatus($saleId, $status) {

        if (!$saleId || !$status) {
            throw new Exception('Sale Id and Status are required');
        }

        $data = [
            'SaleId' => $saleId,
            'Status' => $status,
            'TenantToken' => $this->tenantToken,
            'UserToken' => $this->userToken,
        ];

        $result = $this->remote->call($data, $this->pathPart . '/syncShippedSaleAndRemoveItems');
        return $result;
    }

}