<?php

/*
 * Copyright Talisman Innovations Ltd. (2016). All rights reserved.
 */

namespace SkuVault;

use SkuVault\Utils;
use SkuVault\Exception;
use Psr\Log\LoggerInterface;

class Remote
{

    protected $utils; 
    protected $logger = null;
    protected $header;
    protected $retryCount = 0;

    public function __construct(LoggerInterface $logger)
    {
        $this->utils = new Utils();
        $this->logger = $logger;
    }

    /**
     * 
     * @param type $data
     * @param type $call
     * @return object arrays
     */
    public function call($data, $call)
    {

        $this->retryCount = 0;

        $result = $this->jsonCall($data, $call);

        # json decode the returned body
        return json_decode($result);
    }

    private function jsonCall($data, $call)
    {

        #reset Header
        $this->header = null;

        $result = $this->curl($data, $call);

        # Break response into header and body

        $body = str_replace($this->header, '', $result[0]);

        # Convert response header to array
        $headers = explode("\r\n", $this->header);
        # set default reset time
        $resetLimit = 60;
        # Find the header set reset time
        foreach ($headers as $header) {
            if (strpos($header, 'X-RateLimit-Reset') !== false) {
                // Get the seconds
                $resetLimit = explode(':', $header);
                $seconds = $resetLimit[1] * 3600 + $resetLimit[2] * 60 + $resetLimit[3];
                break;
            }
        }

        # Check for a timeout http code
        # recall if necessary
        switch ($result[1]['http_code']) {
            case 429:
                sleep(round($seconds));
                $body = $this->jsonCall($data, $call);
                break;
            case 0:
                if ($this->retryCount < 3) {
                    $this->logger->debug('HTTP CODE 0 received ' . $this->retryCount . ':', $result);
                    sleep(5);
                    $body = $this->jsonCall($data, $call);
                }
                $this->retryCount++;
                break;
        }

        return $body;
    }

    /**
     * Use the header callback to 
     * write the actual header returned from
     * a call so we can use it to split out the body
     * from the result
     * @param type $ch
     * @param type $header_line
     * @return int
     */
    public function headerCallback($ch, $header_line)
    {

        $this->header .= $header_line;

        return strlen($header_line);
    }

    /**
     * Curl Call
     * @param type $data
     * @param string $call
     * @return type
     */
    protected function curl($data, $call)
    {

        $json = json_encode($data);
        $this->logger->debug('JSON:'. $json);
        
        $ch = curl_init($this->utils->getApiUrl() . $call);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, "POST");
        curl_setopt($ch, CURLOPT_POSTFIELDS, $json);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
        curl_setopt($ch, CURLOPT_TIMEOUT, 120);
        curl_setopt($ch, CURLINFO_HEADER_OUT, true);
        curl_setopt($ch, CURLOPT_HEADER, true);
        curl_setopt($ch, CURLOPT_HEADERFUNCTION, array($this, 'headerCallback'));

        curl_setopt($ch, CURLOPT_HTTPHEADER, array(
            'Content-Type: application/json',
            'Content-Length: ' . strlen($json))
        );

        $result = curl_exec($ch);
        $info = curl_getinfo($ch);
        $error = curl_error($ch);
        
        $this->logger->debug('CURL INFO:', $info);
        $this->logger->debug('CURL  RESULT:'. $result);
        $this->logger->debug('CURL ERROR:'. $error);

        $checkCode = $this->errorCodes($info['http_code']);

        if (!is_null($checkCode)) {
            throw new Exception($checkCode . '  Data: ' . json_encode([$info, $result]));
        }

        return [$result, $info];
    }

    /**
     * Error code check
     * @param type $code
     * @return string
     */
    protected function errorCodes($code)
    {

        switch ($code) {
            case 200:
            case 429:
            case 0:
                $error = null;
                break;
            case 202:
                $error = 'This status is returned when part of a bulk call returns an error, but other elements of the payload are OK.';
                break;
            case 400:
                $error = '400 Bad Request: the call is semantically invalid. It should not be made again without modifications.';
                break;
            case 401:
                $error = '401 Unauthorized: Returned when a disabled user tries to use the API, or your tokens are bad. See Errors in response body.';
                break;
            case 404:
                $error = '404 Page does not exist.';
                break;
            case 409:
                $error = '409 conflict: In bulk calls, this status is returned when two identical, incompatible key:value pairs are provided in a payload. Example: Updating product details, and including sku "product1" twice.';
                break;
            case 413:
                $error = '413 Request Entity Too Large: Returned when the client has attempted to make a bulk call with an array larger than the allowed size.';
                break;
            case 411:
                $error = '411 Length Required';
                break;
            default :
                $error = 'Unknown error';
                break;
        }
        return $error;
    }

}
