<?php

/*
 * Copyright Talisman Innovations Ltd. (2016). All rights reserved.
 * 
 *  $application = new SkuVault\Application('username', 'password');
  $application->getTokens();

  $tenantToken = $application->getTenantToken();
  $userToken = $application->getUserToken();

  $orders = new SkuVault\PurchaseOrders($tenantToken, $userToken);
 */

namespace SkuVault;

use SkuVault\Remote;
use SkuVault\Utils;
use SkuVault\Exception;
use DateTime;
use Psr\Log\LoggerInterface;

class PurchaseOrders {

    protected $remote;
    protected $pathPart;
    protected $utils;
    protected $tenantToken;
    protected $userToken;

    /**
     * Setup
     * @param type $tenantToken
     * @param type $userToken
     */
    public function __construct($tenantToken, $userToken, LoggerInterface $logger) {
        $this->remote = new Remote($logger);
        $this->utils = new Utils();
        $this->pathPart = 'purchaseOrders';
        $this->tenantToken = $tenantToken;
        $this->userToken = $userToken;
    }

    /**
     * Get Purchase Orders
     * @param int $pageNumber
     * @param type $modifiedAfter
     * @param type $modifiedBefore
     * @param string $status
     * @param bool $includeProducts
     * @return json
     * @throws Exception
     */
    public function getPurchaseOrders($pageNumber = null, $modifiedAfter = null, $modifiedBefore = null, $status = null, $includeProducts = null) {

        if ((!is_null($modifiedBefore) && DateTime::createFromFormat('Y-m-d G:i:s', $modifiedBefore) === FALSE) || (!is_null($modifiedAfter) && DateTime::createFromFormat('Y-m-d G:i:s', $modifiedAfter) === FALSE)) {
            throw new Exception('Check the entered date format');
        }

        $data = [];

        if (!is_null($pageNumber)) {
            $data['PageNumber'] = $pageNumber;
        }
        if (!is_null($status)) {
            $data['Status'] = $status;
        }
        if (!is_null($includeProducts)) {
            $data['IncludeProducts'] = $includeProducts;
        }
        if (!is_null($modifiedBefore)) {
            $data['ModifiedBeforeDateTimeUtc'] = $this->utils->formatDate($modifiedBefore);
        }
        if (!is_null($modifiedAfter)) {
            $data['ModifiedAfterDateTimeUtc'] = $this->utils->formatDate($modifiedAfter);
        }

        $data['TenantToken'] = $this->tenantToken;
        $data['UserToken'] = $this->userToken;

        $result = $this->remote->call($data, $this->pathPart . '/getPOs');
        return $result;
    }

    /**
     * Get All Pos depaged
     * Note this can contain two objects PurchaseOrders and Products
     * if include products is selected, it always contains PurchaseOrders
     * @param int $pageNo
     * @return object
     */
    public function getAllPurchaseOrders($modifiedAfter = null, $modifiedBefore = null, $status = null, $includeProducts = null) {

        $items = [];
        $products = [];

        $pageNo = 0;
        while (1) {
            $data = $this->getPurchaseOrders($pageNo, $modifiedAfter, $modifiedBefore, $status, $includeProducts);
            $pageNo++;
            if (empty($data->PurchaseOrders)) {
                break;
            }
            $items = array_merge($items, $data->PurchaseOrders);
            if (isset($data->Products)) {
                $products = array_merge($products, $data->Products);
            }
        }

        $result = (object) ['PurchaseOrders' => $items, 'Products' => $products];

        return $result;
    }

    /**
     * Get Incoming Purchase Orders
     * Get incoming items for incomplete purchase orders
     * @return json
     */
    public function getIncomingItems() {

        $data = [
            'TenantToken' => $this->tenantToken,
            'UserToken' => $this->userToken,
        ];

        $result = $this->remote->call($data, $this->pathPart . '/getIncomingItems');
        return $result;
    }

    /**
     * Create a Purchase Order
     * usage
     * 
     * 
     * use setItem(etc) in an array to add items
     * use setPayment(etc) to add payments
     * 
     * @return json
     * @throws Exception
     */
    public function createPurchaseOrder($SupplierName, $PoNumber = null, $OrderDate = null, $OrderCancelDate = null, $ArrivalDueDate = null, $RequestedShipDate = null, $TrackingInfo = null, $PublicNotes = null, $TermsName = null, $ShippingCarrierClass = null, $LineItems, $Payments = null, $SentStatus = null, $PaymentStatus = null) {


        $order['SupplierName'] = $SupplierName;  // Required (string)
        if ($PoNumber) {
            $order['PoNumber'] = $PoNumber;
        } // (string) auto generated if left blank
        if ($OrderDate) {
            $order['OrderDate'] = $this->utils->formatDate($OrderDate);
        } // (datetime)
        if ($OrderCancelDate) {
            $order['OrderCancelDate'] = $this->utils->formatDate($OrderCancelDate);
        } // (datetime)
        if ($ArrivalDueDate) {
            $order['ArrivalDueDate'] = $this->utils->formatDate($ArrivalDueDate);
        } // (datetime)
        if ($RequestedShipDate) {
            $order['RequestedShipDate'] = $this->utils->formatDate($RequestedShipDate);
        } // (datetime)
        if ($TrackingInfo) {
            $order['TrackingInfo'] = $TrackingInfo;
        } // (string)
        if ($PublicNotes) {
            $order['PublicNotes'] = $PublicNotes;
        } // (string)
        if ($PrivateNotes) {
            $order['PrivateNotes'] = $PrivateNotes;
        } // (string)
        if ($TermsName) {
            $order['TermsName'] = $TermsName;
        } // (string)
        if ($ShippingCarrierClass) {
            $order['ShippingCarrierClass'] = $ShippingCarrierClass;
        } // (object) object ['CarrierName'=>'string', 'ClassName'=>'string']
        $order['LineItems'] = $LineItems; // Required  ['SKU'=>'string', 'Quantity'=>'int', 'QuantityTo3PL'=>'int', 'Cost'=>'double', 'PrivateNotes'=>'string', 'PublicNotes'=>'string', 'Variant'=>'string', 'Identifier'=>'string']
        if ($Payments) {
            $order['Payments'] = $Payments;
        } // (array) of objects ['PaymentName'=>'string', 'Amount'=>'int', 'Note'=>'string']
        if ($SentStatus) {
            $order['SentStatus'] = $SentStatus;
        } // (string) options  NotSent, Sent, and NeedToResend
        if ($PaymentStatus) {
            $order['PaymentStatus'] = $PaymentStatus;
        } // (string) options NonePaid, PartiallyPaid, and FullyPaid

        $order['TenantToken'] = $this->tenantToken;
        $order['UserToken'] = $this->userToken;

        $result = $this->remote->call($data, $this->pathPart . '/createPO');
        return $result;
    }

    public function setItem($sku, $qty, $qty3Pl, $cost, $privateNotes, $publicNotes, $variant, $identifier) {
        return ['SKU' => $sku, 'Quantity' => (int) $qty, 'QuantityTo3PL' => $qty3Pl, 'Cost' => (float) number_format($cost, 2, '.', ''), 'PrivateNotes' => $privateNotes, 'PublicNotes' => $publicNotes, 'Variant' => $variant, 'Identifier' => $identifier];
    }

    public function setPayment($paymentName, $amount, $note) {
        return ['PaymentName' => $paymentName, 'Amount' => (float) number_format($amount, 2, '.', ''), 'Note' => $note];
    }

    /**
     * Receive PO Items
     * 
     * @param string $poNumber
     * @param string $supplierName
     * @param type $receiptDate
     * @param array $lineItems
     * LineItems should be an array of objects
     * 
     * Usage:
     * 
     * $lineItems = [];
     * 
     * for($i=0;$i<10;$i++) {
     * $lineItems[] = $this->setLineItem($sku, $qty, $qtyT3pl);
     * }
     * 
     * $this->receivePOItems($poNumber, $supplierName, $receiptDate, $lineItems);
     * 
     * @return json
     * @throws Exception
     */
    public function receivePOItems(string $poNumber, string $supplierName, $receiptDate, array $lineItems) {

        if (!$poNumber || !$supplierName || !$receiptDate || !$lineItems || empty($lineItems)) {
            throw new Exception('PO Number, Supplier Name, Receipt date and lineitems are required');
        }

        $data = [
            'PoNumber' => $poNumber,
            'SupplierName' => $supplierName,
            'ReceiptDate' => $this->utils->formatDate($receiptDate),
            'LineItems' => $lineItems,
            'TenantToken' => $this->tenantToken,
            'UserToken' => $this->userToken,
        ];

        $result = $this->remote->call($data, $this->pathPart . '/receivePOItems');
        return $result;
    }

    public function setLineItem(string $sku, int $qty, int $qtyT3pl) {
        return ['SKU' => $sku, 'Quantity' => $qty, 'QuantityTo3PL' => $qtyT3pl];
    }

    /**
     * Returns a list of purchase order receives.
     * All dates are returned in UTC format. Therefore, the "ReceiptDate" parameter will almost always be a day ahead of when the actual receipt is. For example, in the above "Example Response," the Receipt Date of "2016-01-23T00:00:00.0000000Z" is actually January 22 (UTC -> EST for example).
     * The "ReceivedDate" returned in the Corrections array will not contain a time. This is because it is impossible for this to be accurate. Why? Because you may have multiple receives at different times for the same SKU on the same day.
     * The Code/PartNumber parameters are the Code/PartNumber on that PO. So, if the product is deleted/recreated, it will not contain the new information, because the old product is what is on the PO.
     * @param date $ReceivedBeforeDateTimeUTC
     * @param date $ReceivedAfterDateTimeUTC
     * @param string $PoNumberFilter
     * @param array $WarehouseFilter [name1, name2]
     * @param int $PageSize
     * @param int $PageNumber
     * @return object
     */
    public function getReceivesHistory($ReceivedBeforeDateTimeUTC = null, $ReceivedAfterDateTimeUTC = null, $PoNumberFilter = null, $WarehouseFilter = null, $PageSize = 10000, $PageNumber = null) {

        $data = [];

        if ($ReceivedBeforeDateTimeUTC) {
            $data['ReceivedBeforeDateTimeUTC'] = $this->utils->formatDate($ReceivedBeforeDateTimeUTC);
        }
        if ($ReceivedBeforeDateTimeUTC) {
            $data['ReceivedAfterDateTimeUTC'] = $this->utils->formatDate($ReceivedAfterDateTimeUTC);
        }
        if ($PoNumberFilter) {
            $data['PoNumberFilter'] = $PoNumberFilter;
        }
        if ($WarehouseFilter) {
            $data['WarehouseFilter'] = $WarehouseFilter;
        }
        $data['PageSize'] = $PageSize;
        if ($PageNumber) {
            $data['PageNumber'] = $PageNumber;
        }
        $data['TenantToken'] = $this->tenantToken;
        $data['UserToken'] = $this->userToken;

        $result = $this->remote->call($data, $this->pathPart . '/getReceivesHistory');
        return $result;
    }

    /**
     * Depaged getReceivesHistory
     * @param date $ReceivedBeforeDateTimeUTC
     * @param date $ReceivedAfterDateTimeUTC
     * @param string $PoNumberFilter
     * @param array $WarehouseFilter [name1, name2]
     * @param int $PageSize
     * @return object
     */
    public function getAllReceivesHistory($ReceivedBeforeDateTimeUTC = null, $ReceivedAfterDateTimeUTC = null, $PoNumberFilter = null, $WarehouseFilter = null, $PageSize = 10000) {

        $items = [];
        $pageNo = 0;
        while (1) {
            $data = $this->getReceivesHistory($ReceivedBeforeDateTimeUTC, $ReceivedAfterDateTimeUTC, $PoNumberFilter, $WarehouseFilter, $PageSize, $pageNo);
            $pageNo++;
            if (empty($data->Receives)) {
                break;
            }
            $items = array_merge($items, $data->Receives);
        }

        $result = (object) ['Receives' => $items];

        return $result;
    }

}
