<?php

/*
 * Copyright Talisman Innovations Ltd. (2016). All rights reserved.
 * 
 *  $application = new SkuVault\Application('username', 'password');
  $application->getTokens();

  $tenantToken = $application->getTenantToken();
  $userToken = $application->getUserToken();

  $products = new SkuVault\Products($tenantToken, $userToken);
 * 
 * 
 */

namespace SkuVault;

use SkuVault\Remote;
use SkuVault\Utils;
use SkuVault\Exception;
use DateTime;
use Psr\Log\LoggerInterface;

class Products {

    protected $remote;
    protected $pathPart;
    protected $utils;
    protected $tenantToken;
    protected $userToken;
    protected $logger;

    /**
     * Setup
     * @param type $tenantToken
     * @param type $userToken
     */
    public function __construct($tenantToken, $userToken, LoggerInterface $logger) {
  
        $this->remote = new Remote($logger);
        $this->utils = new Utils();
        $this->pathPart = 'products';
        $this->tenantToken = $tenantToken;
        $this->userToken = $userToken;
    }

    /**
     * Get Suppliers
     * @param int $pageNumber
     * @return json
     */
    public function getSuppliers($pageNumber=null) {

        $data = [];
        if (!is_null($pageNumber)) {
            $data['PageNumber'] = $pageNumber;
        }
        $data['TenantToken'] = $this->tenantToken;
        $data['UserToken'] = $this->userToken;

        $result = $this->remote->call($data, $this->pathPart . '/getSuppliers');
        return $result;
    }

     /**
     * Get All Suppliers depaged
     * @param int $pageNumber
     * @return object
     */
    public function getAllSuppliers() {

       $items = [];

        $pageNo = 0;
        while (1) {
            $data = $this->getSuppliers($pageNo);
            $pageNo++;
            if (empty($data->Suppliers)) {
                break;
            }
            $items = array_merge($items, $data->Suppliers);
        }
        
        $result = (object)['Suppliers'=>$items];

        return $result;
    }
    
    
    /**
     * Create Suppliers
     * Pass in an array of suppliers
     * Use the create Supplier method
     * 
     * $array = [];
     * $emails = ['email1','email2'];'
     *
     * 
     * for($i=0;$i<10;$i++) {
     * $array[] = $this->createSupplier($name, $subject, $templatel, $message, $emails]);
     * }
     * 
     * $this->createSuppliers($array);
     * 
     * @param array $suppliers
     * @return json
     */
    public function createSuppliers($suppliers) {

        if (!$suppliers || !empty($suppliers)) {
            throw new Exception('Suppliers array required');
        }

        $data = [
            'Suppliers' => $suppliers,
            'TenantToken' => $this->tenantToken,
            'UserToken' => $this->userToken,
        ];

        $result = $this->remote->call($data, $this->pathPart . '/createSuppliers');
        return $result;
    }

    public function createSupplier(string $name = null, string $subject = null, string $template = null, string $message = null, array $emails = []) {
        return ['Name' => $name, 'EmailTemplateSubject' => $subject, 'EmailTemplateMessage' => $template, 'Emails' => $emails];
    }

    /**
     * Get Brands
     * @param int $pageNumber
     * @return object
     */
    public function getBrands($pageNumber = null) {

        $data = [];
        if (!is_null($pageNumber)) {
            $data['PageNumber'] = $pageNumber;
        }
        $data['TenantToken'] = $this->tenantToken;
        $data['UserToken'] = $this->userToken;

        $result = $this->remote->call($data, $this->pathPart . '/getBrands');
        return $result;
    }
    
    /**
     * Get All Brands depaged
     * @param int $pageNumber
     * @return object
     */
    public function getAllBrands() {

       $items = [];

        $pageNo = 0;
        while (1) {
            $data = $this->getBrands($pageNo);
            $pageNo++;
            if (empty($data->Brands)) {
                break;
            }
            $items = array_merge($items, $data->Brands);
        }
        
        $result = (object)['Brands'=>$items];

        return $result;
    }

    /**
     * Create Brands
     * Pass in an array of brands
     * [['Name':'brand1'], ['Name':'brand2']]
     * @param array $brands
     * @return json
     */
    public function createBrands($brands) {

        if (!$brands || !empty($brands)) {
            throw new Exception('Brands array required');
        }

        $data = [
            'Brands' => $brands,
            'TenantToken' => $this->tenantToken,
            'UserToken' => $this->userToken,
        ];

        $result = $this->remote->call($data, $this->pathPart . '/createBrands');
        return $result;
    }

    /**
     * Get Classifications
     * @param int $pageNumber
     * @return json
     */
    public function getClassifications($pageNumber = null) {

        $data = [];
        if (!is_null($pageNumber)) {
            $data['PageNumber'] = $pageNumber;
        }
        $data['TenantToken'] = $this->tenantToken;
        $data['UserToken'] = $this->userToken;

        $result = $this->remote->call($data, $this->pathPart . '/getClassifications');
        return $result;
    }
    
    /**
     * Get All Classifications depaged
     * @param int $pageNumber
     * @return object
     */
    public function getAllClassifications() {

       $items = [];

        $pageNo = 0;
        while (1) {
            $data = $this->getClassifications($pageNo);
            $pageNo++;
            if (empty($data->Classifications)) {
                break;
            }
            $items = array_merge($items, $data->Classifications);
        }
        
        $result = (object)['Classifications'=>$items];
        
        return $result;
    }
    
    
     /**
     * Get all Products
     * depaged results
     * @param type $modifiedAfter
     * @param type $modifiedBefore
     * @param bool $skus
     * @return object
     */
    public function getAllProducts($modifiedAfter = null, $modifiedBefore = null, $skus = null) {

        $items = [];
        $pageSize = 10000;

        $pageNo = 0;
        while (1) {
            $products = $this->getProducts($pageSize, $pageNo, $modifiedAfter, $modifiedBefore, $skus);
            $pageNo++;
            if (empty($products->Products)) {
                break;
            }
            $items = array_merge($items, $products->Products);
        }
        
        $result = (object)['Products'=>$items];

        return $result;
    }
    

    /**
     * Get Products
     * @param int $pageSize
     * @param int $pageNumber
     * @param datetime $modifiedAfter
     * @param datetime $modifiedBefore
     * @param string $skus
     * @return json
     */
    public function getProducts($pageSize = null, $pageNumber = null, $modifiedAfter = null, $modifiedBefore = null, $skus = null) {

        if ((!is_null($modifiedBefore) && DateTime::createFromFormat('Y-m-d G:i:s', $modifiedBefore) === FALSE) || (!is_null($modifiedAfter) && DateTime::createFromFormat('Y-m-d G:i:s', $modifiedAfter) === FALSE)) {
            throw new Exception('Check the entered date format');
        }

        $data = [];

        if (!is_null($modifiedAfter)) {
            $data['ModifiedAfterDateTimeUtc'] = $this->utils->formatDate($modifiedAfter);
        }
        if (!is_null($modifiedBefore)) {
            $data['ModifiedBeforeDateTimeUtc'] = $this->utils->formatDate($modifiedBefore);
        }
        
        if (!is_null($skus)) {
            $data['Skus'] = $skus;
        }
        if (!is_null($pageNumber)) {
            $data['PageNumber'] = $pageNumber;
        }
        if (!is_null($pageSize)) {
            $data['PageSize'] = $pageSize;
        }

        $data['TenantToken'] = $this->tenantToken;
        $data['UserToken'] = $this->userToken;
                
        $result = $this->remote->call($data, $this->pathPart . '/getProducts');
        
        return $result;
    }

    /**
     * Get Product By Sku
     * 
     * @param string $sku
     * @return json
     */
    public function getProductBySku(string $sku) {
        $data = [];
        $data['TenantToken'] = $this->tenantToken;
        $data['UserToken'] = $this->userToken;
        $data['ProductSKUs'] = [$sku];

        $result = $this->remote->call($data, $this->pathPart . '/getProducts');
        return $result;
    }

    public function setProduct($Sku, $Description = null, $ShortDescription = null, $LongDescription = null, $Classification, $Supplier, $Brand, $Code = null, $PartNumber, $Cost, $SalePrice, $RetailPrice, $Weight, $UnitWeight = 'kgs', $VariationParentSku, $ReorderPoint, $Pictures, $Note, $MinimumOrderQuantity, $MinimumOrderQuantityInfo, $Statuses, $Attributes
    ) {

        $product = [];
        $product[] = $Sku; // Required (string)
        if ($Description) {
            $product['Description'] = $Description;
        } // alias Title (string)
        if ($ShortDescription) {
            $product['ShortDescription'] = $ShortDescription;
        } // (string 1000 char max)
        if ($LongDescription) {
            $product['LongDescription'] = $LongDescription;
        } // (string 65665 char max)
        $product['Classification'] = $Classification;  // Required (Select from Skuvault)
        $product['Supplier'] = $Supplier;  // Required (Select from Skuvault)
        $product['Brand'] = $Brand;  // Required (Select from Skuvault)
        if ($Code) {
            $product['Code'] = $Code;
        } // Barcode number (string)
        if ($PartNumber) {
            $product['PartNumber'] = $PartNumber;
        } // (string)
        if ($Cost) {
            $product['Cost'] = (float) number_format($Cost, 2, '.', '');
        } // (decimal)
        if ($SalePrice) {
            $product['SalePrice'] = (float) number_format($SalePrice, 2, '.', '');
        } // (decimal)
        if ($RetailPrice) {
            $product['RetailPrice'] = (float) number_format($RetailPrice, 2, '.', '');
        } // (decimal)
        if ($Weight) {
            $product['Weight'] = (float) number_format($Weight, 2, '.', '');
        } // (decimal/double)
        if ($UnitWeight) {
            $product['UnitWeight'] = $UnitWeight;
        } // ['lbs','kgs', 'oz', 'g']
        if ($VariationParentSku) {
            $product['VariationParentSku'] = $VariationParentSku;
        } // (string)
        if ($ReorderPoint) {
            $product['ReorderPoint'] = (int)$ReorderPoint;
        } // (int)
        if ($Pictures) {
            $product['Pictures'] = $Pictures;
        } // (array) list if image urls['url','url']
        if ($AllowCreateAp) {
            $product['AllowCreateAp'] = $AllowCreateAp;
        } // (boolean) Creates an Assembled Product if the kit already exists.
        if ($Note) {
            $product['Note'] = $Note;
        } // (string)
        if ($MinimumOrderQuantity) {
            $product['MinimumOrderQuantity'] = (int)$MinimumOrderQuantity;
        } // (int)
        if ($MinimumOrderQuantityInfo) {
            $product['MinimumOrderQuantityInfo'] = (int)$MinimumOrderQuantityInfo;
        } // (string)
        if ($Statuses) {
            $product['Statuses'] = $Statuses;
        } // (array) ['string', 'string']
        if ($Attributes) {
            $product['Attributes'] = $Attributes;
        } // (array in array) ie [ ['AttributeName'=>'AttributeValue','AttributeName'=>'AttributeValue'] ]

        return $product;
    }

    /**
     * Create Product
     * $product = $this->setProduct;
     * $this->createProduct($product);
     * 
     * @return json
     * $usage
     * $product = $this->setProduct(etc);

     */
    public function createProduct($product) {

        if (!$product || empty($product)) {
            throw new Exception('Product Object required');
        }

        $product['TenantToken'] = $this->tenantToken;
        $product['UserToken'] = $this->userToken;

        $result = $this->remote->call($data, $this->pathPart . '/createProduct');
        return $result;
    }

    /**
     * Create products
     * $products = [];
     * $products[] = $this->setProduct(etc);
     * $this->createProducts($products);
     * @return json
     * @throws Exception
     */
    public function createProducts($products) {

        if (!$products || empty($products)) {
            throw new Exception('No Products passed in');
        }

        $data = [
            $products,
            'TenantToken' => $this->tenantToken,
            'UserToken' => $this->userToken,
        ];

        $result = $this->remote->call($data, $this->pathPart . '/createProducts');
        return $result;
    }

    /**
     * Update product
     * $product = $this->setProduct;
     * $this->createProduct($product);
     * @param object $product
     * @return json
     */
    public function updateProduct($product) {

        if (!$product || empty($product)) {
            throw new Exception('Product Object required');
        }

        $product['TenantToken'] = $this->tenantToken;
        $product['UserToken'] = $this->userToken;

        $result = $this->remote->call($data, $this->pathPart . '/updateProduct');

        return $result;
    }

    /**
     * Update Product
     * @param array $products contains product 
     * @return json
     */
    public function updateProducts(array $products) {

        if (!$products || empty($products)) {
            throw new Exception('No Products passed in');
        }

        $data = [
            $products,
            'TenantToken' => $this->tenantToken,
            'UserToken' => $this->userToken,
        ];

        $result = $this->remote->call($data, $this->pathPart . '/updateProducts');
        return $result;
    }

    /**
     * Get Kits
     * @param UTC date (2016-01-12T00:00:00.0000000Z) $modifiedAfter
     * @param UTC date (2016-01-12T00:00:00.0000000Z) $modifiedBefore
     * @param array $skus
     * @param int $pageNumber
     * @param boolean $includeKitCost
     * @return json
     */
    public function getKits($modifiedAfter = null, $modifiedBefore = null, array $skus = null, $pageSize = null, $pageNumber = null, $includeKitCost = null) {

        if ((!is_null($modifiedBefore) && DateTime::createFromFormat('Y-m-d G:i:s', $modifiedBefore) === FALSE) || (!is_null($modifiedAfter) && DateTime::createFromFormat('Y-m-d G:i:s', $modifiedAfter) === FALSE)) {
            throw new Exception('Check the entered date format');
        }

        $data = [];

        if (!is_null($pageNumber)) {
            $data['PageNumber'] = $pageNumber;
        }
        if (!is_null($pageSize)) {
            $data['PageSize'] = $pageSize;
        }
        if (!is_null($skus)) {
            $data['KitSKUs'] = $skus;
        }
        if (!is_null($includeKitCost)) {
            $data['IncludeKitCost'] = $includeKitCost;
        }
        if (!is_null($modifiedBefore)) {
            $data['AvailableQuantityModifiedBeforeDateTimeUtc'] = $this->utils->formatDate($modifiedBefore);
        }
        if (!is_null($modifiedAfter)) {
            $data['AvailableQuantityModifiedAfterDateTimeUtc'] = $this->utils->formatDate($modifiedAfter);
        }

        $data['TenantToken'] = $this->tenantToken;
        $data['UserToken'] = $this->userToken;

        $result = $this->remote->call($data, $this->pathPart . '/getKits');
        return $result;
    }

    /**
     * Create Kit
     * 
     * $items = ['sku1', 'sku2', 'etc'];
     * $kitlines = [];
     * $sku = 'kit1';
     * $title = 'Title 1';
     * 
     * for($i=0;$i<10;$i++) {
     * $array[] = $this->setKitLine($name, $combine, $qty, $items);
     * }
     * 
     * $this->createKits($sku, $title, '', '', '$kitlines);
     * 
     * @param string $sku
     * @param string $title
     * @param string $code
     * @param boolean $allowCreateAp
     * @param array $kitlines containing array ie [['LineName'=>'LineName', 'Combine'=>'Combine', 'Quantity'=>2, 'Items'=>['sku1', 'sku2']]
     * @return json
     */
    public function createKits($sku, $title, $code = null, $allowCreateAp = null, $kitlines) {

        if (!$sku || !$title || !$kitlines || empty($kitlines)) {
            throw new Exception('Sku, Title and Kitlines array required');
        }

        $data = [];

        if (!is_null($code)) {
            $data['Code'] = $code;
        }
        if (!is_null($allowCreateAp)) {
            $data['AllowCreateAp'] = $allowCreateAp;
        }

        $data['Sku'] = $sku;
        $data['Title'] = $title;
        $data['KitLines'] = $kitlines;
        $data['TenantToken'] = $this->tenantToken;
        $data['UserToken'] = $this->userToken;

        $result = $this->remote->call($data, $this->pathPart . '/createKit');
        return $result;
    }

    public function setKitLine($name, $combine, $qty, $items) {
        return ['LineName' => $name, 'Combine' => $combine, 'Quantity' => $qty, 'Items' => $items];
    }

}
