<?php

/* 
 * Copyright Talisman Innovations Ltd. (2016). All rights reserved.
 * 
 *  $application = new SkuVault\Application('username', 'password');

    $tenantToken = $application->getTenantToken();
    $userToken = $application->getUserToken();

    $inventory = new SkuVault\Inventory($tenantToken, $userToken);
 * 
 * 
 */

namespace SkuVault;

use SkuVault\Remote;
use SkuVault\Utils;
use SkuVault\Exception;
use DateTime;
use Psr\Log\LoggerInterface;

class Inventory {
    
    protected $remote;
    protected $pathPart;
    protected $utils;
    protected $tenantToken;
    protected $userToken;

    /**
     * Setup
     * @param type $tenantToken
     * @param type $userToken
     */
    public function __construct($tenantToken, $userToken, LoggerInterface $logger) {
        $this->remote = new Remote($logger);
        $this->utils = new Utils();
        $this->pathPart = 'inventory';
        $this->tenantToken = $tenantToken;
        $this->userToken = $userToken;
    }
    
    /**
     * Get all Available quantities 
     * depaged results
     * @param type $modifiedBeforeDateTimeUtc
     * @param type $modifiedAfterDateTimeUtc
     * @param bool $expandAlternateSkus
     * @return object
     */
    public function getAllAvailableQuantities($modifiedBeforeDateTimeUtc = null, $modifiedAfterDateTimeUtc = null, bool $expandAlternateSkus = null) {

        $items = [];
        $pageSize = 5000;

        $pageNo = 0;
        while (1) {
            $availableQty = $this->getAvailableQuantities($pageNo, $pageSize, $modifiedBeforeDateTimeUtc, $modifiedAfterDateTimeUtc, $expandAlternateSkus);
            $pageNo++;
            if (empty($availableQty->Items)) {
                break;
            }
            $items = array_merge($items, $availableQty->Items);
        }
        
        $result = (object)['Items'=>$items];
        
        return $result;
    }

    /**
     * Get Available Quantities
     * 
     * @param int $pageNumber
     * @param datetime $modifiedBeforeDateTimeUtc
     * @param datetime $modifiedAfterDateTimeUtc
     * @param bool $expandAlternateSkus
     * @return object
     * @throws Exception
     */
    public function getAvailableQuantities($pageNumber = null, $pageSize = null, $modifiedBeforeDateTimeUtc = null, $modifiedAfterDateTimeUtc = null, $expandAlternateSkus = null) {

        if ((!is_null($modifiedBeforeDateTimeUtc) && DateTime::createFromFormat('Y-m-d G:i:s', $modifiedBeforeDateTimeUtc) === FALSE) 
                || (!is_null($modifiedAfterDateTimeUtc) && DateTime::createFromFormat('Y-m-d G:i:s', $modifiedAfterDateTimeUtc) === FALSE)) {
            throw new Exception('Check the entered date format');
        }

        $data = [];
        
        if(!is_null($pageNumber)) {
            $data['PageNumber'] = $pageNumber;
        }
        
        if(!is_null($pageSize)) {
            $data['PageSize'] = $pageSize;
        }
        
        if(!is_null($modifiedBeforeDateTimeUtc)) {
            $data['ModifiedBeforeDateTimeUtc'] = $this->utils->formatDate($modifiedBeforeDateTimeUtc);
        }
        if(!is_null($modifiedAfterDateTimeUtc)) {
            $data['ModifiedAfterDateTimeUtc'] = $this->utils->formatDate($modifiedAfterDateTimeUtc);
        }
        if(!is_null($expandAlternateSkus)) {
            $data['ExpandAlternateSkus'] = $expandAlternateSkus;
        }
        
        $data['TenantToken'] = $this->tenantToken;
        $data['UserToken'] = $this->userToken;

        $result = $this->remote->call($data, $this->pathPart . '/getAvailableQuantities');
        
        return $result;
    }
    
    /**
     * Get Product/Item Quantities 
     * 
     * @param int $pageNumber
     * @param int $pageSize
     * @param datetime $modifiedBeforeDateTimeUtc
     * @param datetime $modifiedAfterDateTimeUtc
     * @param array $productCodes
     * @return object
     * @throws Exception
     */
    public function getItemQuantities($pageNumber = null, $pageSize = null, $modifiedBeforeDateTimeUtc = null, $modifiedAfterDateTimeUtc = null, $productCodes = null) {
        
        if ((!is_null($modifiedBeforeDateTimeUtc) && DateTime::createFromFormat('Y-m-d G:i:s', $modifiedBeforeDateTimeUtc) === FALSE) 
                || (!is_null($modifiedAfterDateTimeUtc) && DateTime::createFromFormat('Y-m-d G:i:s', $modifiedAfterDateTimeUtc) === FALSE)) {
            throw new Exception('Check the entered date format');
        }
        
        if(($pageSize && $pageSize < 1000) || ($pageSize && $pageSize > 10000)) {
             throw new Exception('Page size must be between 1000 and 10000');
        }
        
        // Set Up data for json call
        $data = [];
        
        if(!is_null($pageNumber)) {
            $data['PageNumber'] = $pageNumber;
        }
        if(!is_null($modifiedBeforeDateTimeUtc)) {
            $data['ModifiedBeforeDateTimeUtc'] = $this->utils->formatDate($modifiedBeforeDateTimeUtc);
        }
        if(!is_null($modifiedAfterDateTimeUtc)) {
            $data['ModifiedAfterDateTimeUtc'] = $this->utils->formatDate($modifiedAfterDateTimeUtc);
        }
        if(!is_null($productCodes)) {
            $data['ProductCodes'] = $productCodes;
        }
        if(!is_null($pageSize)) {
            $data['PageSize'] = $pageSize;
        }
        
        $data['TenantToken'] = $this->tenantToken;
        $data['UserToken'] = $this->userToken;
       
       
        $result = $this->remote->call($data, $this->pathPart . '/getItemQuantities');
        return $result;
        
    }
    
    /**
     * Get Kit Quantities
     * 
     * @param int $pageNumber
     * @param datetime $modifiedBeforeDateTimeUtc
     * @param datetime $modifiedAfterDateTimeUtc
     * @return json
     * @throws Exception
     */
    public function getKitQuantities($pageNumber = null, $modifiedBeforeDateTimeUtc = null, $modifiedAfterDateTimeUtc = null) {
        
        if ((!is_null($modifiedBeforeDateTimeUtc) && DateTime::createFromFormat('Y-m-d G:i:s', $modifiedBeforeDateTimeUtc) === FALSE) 
                || (!is_null($modifiedAfterDateTimeUtc) && DateTime::createFromFormat('Y-m-d G:i:s', $modifiedAfterDateTimeUtc) === FALSE)) {
            throw new Exception('Check the entered date format');
        }
        
        $data = [];
        
        if(!is_null($pageNumber)) {
            $data['PageNumber'] = $pageNumber;
        }
        if(!is_null($modifiedBeforeDateTimeUtc)) {
            $data['ModifiedBeforeDateTimeUtc'] = $this->utils->formatDate($modifiedBeforeDateTimeUtc);
        }
        if(!is_null($modifiedAfterDateTimeUtc)) {
            $data['ModifiedAfterDateTimeUtc'] = $this->utils->formatDate($modifiedAfterDateTimeUtc);
        }
        
        $data['TenantToken'] = $this->tenantToken;
        $data['UserToken'] = $this->userToken;
        
        $result = $this->remote->call($data, $this->pathPart . '/getKitQuantities');
        return $result;
    }
    
    /**
     * Get All KitQuantities depaged
     * @param int $pageNumber
     * @return object
     */
    public function getAllKitQuantities($modifiedBeforeDateTimeUtc = null, $modifiedAfterDateTimeUtc = null) {

       $items = [];

        $pageNo = 0;
        while (1) {
            $data = $this->getKitQuantities($pageNo, $modifiedBeforeDateTimeUtc, $modifiedAfterDateTimeUtc);
            $pageNo++;
            if (empty($data->Kits)) {
                break;
            }
            $items = array_merge($items, $data->Kits);
        }
        
        $result = (object)['Kits'=>$items];

        return $result;
    }
    
    
    /**
     * Get Warehouse Item Quantity
     * Returns the quantity for a specified SKU.
     * @param string $sku
     * @param int $warehouseId
     * @return object
     * @throws Exception
     */
    public function getWarehouseItemQuantity($sku, $warehouseId) {
        
        if(!$sku || !$warehouseId) {
            throw new Exception('Sku and Warehouse Id are required');
        }
        
        $data = [
            'Sku' => $sku,
            'WarehouseId' => $warehouseId,
            'TenantToken' => $this->tenantToken,
            'UserToken' => $this->userToken,
        ];
        
        $result = $this->remote->call($data, $this->pathPart . '/getWarehouseItemQuantity');
        return $result;
        
    }
    
    /**
     * Get Warehouse Item/Product Quantities
     * This call returns SKUs and quantities from a specified warehouse.
     * @param int $pageNumber
     * @param int $pageSize
     * @param int $warehouseId
     * @return object
     * @throws Exception
     */
    public function getWarehouseItemQuantities($pageNumber = null, $pageSize = null, $warehouseId) {
               
        if(!$warehouseId) {
            throw new Exception('Warehouse Id is required');
        }
        
        if($pageSize && $pageSize < 1000 || $pageSize > 10000) {
             throw new Exception('Page size must be between 1000 and 10000');
        }
        
        $data = [];
        
        if(!is_null($pageNumber)) {
            $data['PageNumber'] = $pageNumber;
        }
        if(!is_null($pageSize)) {
            $data['PageSize'] = $pageSize;
        }
        
        $data['WarehouseId'] = $warehouseId;
        $data['TenantToken'] = $this->tenantToken;
        $data['UserToken'] = $this->userToken;
        
        $result = $this->remote->call($data, $this->pathPart . '/getWarehouseItemQuantities');
        return $result;
        
    }
    
    /**
     * Get all Warehouse quantities 
     * depaged results
     * @param type $modifiedBeforeDateTimeUtc
     * @param type $modifiedAfterDateTimeUtc
     * @param bool $expandAlternateSkus
     * @return object
     */
    public function getAllWarehouseItemQuantities($warehouseId = null) {

        $items = [];
        $pageSize = 10000;

        $pageNo = 0;
        while (1) {
            $warehouseQty = $this->getWarehouseItemQuantities($pageNo, $pageSize, $warehouseId);
            $pageNo++;
            if (empty($warehouseQty->ItemQuantities)) {
                break;
            }
            $items = array_merge($items, $warehouseQty->ItemQuantities);
        }
        
        $result = (object)['ItemQuantities'=>$items];
        
        return $result;
    }

    /**
     * Get Inventory By Location
     * Product locatons
     * @param int $pageNumber
     * @param array $skus
     * @return object
     */
    public function getInventoryByLocation($pageNumber = null, $skus = null) {
        
        $data = [];
        
        $skusArray = array_unique($skus);
        
        if($skusArray && !empty($skus)) {
            $data['ProductSKUs'] = $skusArray;
        }
        if($pageNumber) {
            $data['PageNumber'] = $pageNumber;
        }
        
        $data['TenantToken'] = $this->tenantToken;
        $data['UserToken'] = $this->userToken;
        
        $result = $this->remote->call($data, $this->pathPart . '/getInventoryByLocation');
        return $result;
        
    }
    
      /**
     * Get All InventoryByLocation depaged
     * @param int $pageNo
     * @return object
     */
    public function getAllInventoryByLocation($skus = null) {

       $items = [];

        $pageNo = 0;
        while (1) {
            $data = $this->getInventoryByLocation($pageNo, $skus);
            $pageNo++;
            $getItems = (array)$data->Items;
            if (empty($getItems)) {
                break;
            }
            $items = array_merge($items, $getItems);
        }
        
        $result = (object)['Items'=>(object)$items];

        return $result;
    }
    
    /**
     * Pick Item/Product
     * 
     * @param int $warehouseId
     * @param int $quantity
     * @param string $locationCode
     * @param string $sku
     * @param boolean $isExpressPick
     * @param string $note
     * @return object
     * @throws Exception
     */
    public function pickItem($warehouseId, $quantity, $locationCode = null, $sku = null, $isExpressPick = null, $note = null) {

        if (!$quantity || !$warehouseId) {
            throw new Exception('Quantity and Warehouse Id are required');
        }

        $data = [];
        
        if(!is_null($locationCode)) {
            $data['LocationCode'] = $locationCode;
        }
        if(!is_null($code)) {
            $data['Code'] = $code;
        }
        if(!is_null($sku)) {
            $data['Sku'] = $sku;
        }
        if(!is_null($isExpressPick)) {
            $data['IsExpressPick'] = $isExpressPick;
        }
        if(!is_null($note)) {
            $data['Note'] = $note;
        }
        
        $data['Quantity'] = $quantity;
        $data['WarehouseId'] = $warehouseId;
        $data['TenantToken'] = $this->tenantToken;
        $data['UserToken'] = $this->userToken;

        $result = $this->remote->call($data, $this->pathPart . '/pickItem');
        return $result;
    }
    
    /**
     * Add Quantity/Item
     * Add quantity for a SKU to a warehouse location.
     * @param int $warehouseId
     * @param int $quantity
     * @param string $locationCode
     * @param string $reason
     * @param string $code
     * @param string $sku
     * @return object
     * @throws Exception
     */
    public function addItem($warehouseId, $quantity, $locationCode, $reason, $code = null, $sku = null) {
        
        if (!$quantity || !$warehouseId || !$locationCode || !$reason) {
            throw new Exception('Quantity, Warehouse Id, Location code and Reason are required');
        }

        $data = [];
        
        if(!is_null($sku)) {
            $data['Sku'] = $sku;
        }
        if(!is_null($code)) {
            $data['Code'] = $code;
        }
     
        $data['Reason'] = $reason;
        $data['LocationCode'] = $locationCode;
        $data['Quantity'] = $quantity;
        $data['WarehouseId'] = $warehouseId;
        $data['TenantToken'] = $this->tenantToken;
        $data['UserToken'] = $this->userToken;

        $result = $this->remote->call($data, $this->pathPart . '/addItem');
        return $result;
        
    }
    
    /**
     * Add items in bulk
     *
     * $items = [];
     * 
     * for($i=0;$i<10;$i++) {
     * $items[] = $this->setItem(string $sku, $code, $warehouseId, $locationCode, $qty, $reason);
     * }
     * 
     * $this->addItemBulk($items);
     * 
     * @param array $items
     * @return object
     * @throws Exception
     */
    public function addItemBulk($items) {
        
        if(!$items || empty($items)) {
            throw new Exception('Items array is required');
        }
        
        $data = [
            'Items' => $items,
            'TenantToken' => $this->tenantToken,
            'UserToken' => $this->userToken,
        ];
        
        
        $result = $this->remote->call($data, $this->pathPart . '/addItemBulk');
        
        return $result;
    }
    
    public function setItem($sku=null, $code=null, $warehouseId, $locationCode, $qty, $reason) {
        
        if(!$sku && !$code) {
            throw new Exception('Either Sku or Code is required');
        }
        
        $data = [];
        if($sku) {
            $data['Sku'] = $sku;
        }
        if($code) {
            $data['Code'] = $code;
        }
        $data['WarehouseId'] = $warehouseId;
        $data['LocationCode'] = $locationCode;
        $data['Quantity'] = $qty;
        $data['Reason'] = $reason;
        
        return $data;
    }
    
    
    /**
     * Remove Product/Item Quantity
     * 
     * @param int $warehouseId
     * @param int $quantity
     * @param string $locationCode
     * @param string $reason
     * @param string $code
     * @param string $sku
     * @return object
     * @throws Exception
     */
    public function removeItem($warehouseId, $quantity, $locationCode, $reason, $code = null, $sku = null) {
        
        if (!$quantity || !$warehouseId || !$locationCode || !$reason) {
            throw new Exception('Quantity, Warehouse Id, Location code and Reason are required');
        }

        $data = [];
        
        if(!is_null($sku)) {
            $data['Sku'] = $sku;
        }
        if(!is_null($code)) {
            $data['Code'] = $code;
        }
     
        $data['Reason'] = $reason;
        $data['LocationCode'] = $locationCode;
        $data['Quantity'] = $quantity;
        $data['WarehouseId'] = $warehouseId;
        $data['TenantToken'] = $this->tenantToken;
        $data['UserToken'] = $this->userToken;

        $result = $this->remote->call($data, $this->pathPart . '/removeItem');
        return $result;
    }
    
    /**
     * Remove Item Qty in Bulk
     * 
     * 
     * $items = [];
     * 
     * for($i=0;$i<10;$i++) {
     * $items[] = $this->setItem(string $sku, $code, $warehouseId, $locationCode, $qty, $reason);
     * }
     * 
     * $this->addItemBulk($items);
     * @param array $items
     * @return json
     * @throws Exception
     */
    public function removeItemBulk($items) {
        
        if(!$items || !empty($items)) {
            throw new Exception('Items array is required');
        }
        
        $data = [
            'Items' => $items,
            'TenantToken' => $this->tenantToken,
            'UserToken' => $this->userToken,
        ];
        
        $result = $this->remote->call($data, $this->pathPart . '/removeItemBulk');
        return $result;
    }
    
    /**
     * Set Item Quantity 
     * This lets you explicitly set quantity for an item in a warehouse's location.
     * @param int $warehouseId
     * @param int $quantity
     * @param string $locationCode
     * @param string $code
     * @param string $sku
     * @return json
     * @throws Exception
     */
    public function setItemQuantity($warehouseId, $quantity, s$locationCode, $code = null, $sku = null) {
        
        if (!$quantity || !$warehouseId || !$locationCode) {
            throw new Exception('Quantity, Warehouse Id, Location and Code are required');
        }

        $data = [];
        
        if(!is_null($sku)) {
            $data['Sku'] = $sku;
        }
        if(!is_null($code)) {
            $data['Code'] = $code;
        }
     
        $data['LocationCode'] = $locationCode;
        $data['Quantity'] = $quantity;
        $data['WarehouseId'] = $warehouseId;
        $data['TenantToken'] = $this->tenantToken;
        $data['UserToken'] = $this->userToken;
        
        $result = $this->remote->call($data, $this->pathPart . '/setItemQuantity');
        return $result;
        
    }
    
    /**
     * Get the Warehouses
     * @param int $pageNo
     * @return json
     */
     
    public function getWarehouses($pageNo=null) {

        $data = [];
        if(!is_null($pageNo)) {
            $data['pageNo'] = $pageNo;
        }
        $data['TenantToken'] = $this->tenantToken;
        $data['UserToken'] = $this->userToken;

        return $this->remote->call($data, $this->pathPart .'/getWarehouses');
    }

    //
    
            
    /**
     * Get Transactions History
     * @param datetime $FromDate
     * @param datetime $ToDate
     * @param int $PageNumber
     * @param int $PageSize
     * @param string $TransactionType - All, Add, Remove, or Pick
     * @param array $TransactionReasons
     * @param array $ExcludeTransactionReasons
     * @param int $WarehouseID
     * @return json
     * @throws Exception
     */
    public function getTransactions($FromDate, $ToDate, $PageNumber = null, $PageSize = null, $TransactionType = null, $TransactionReasons = null, $ExcludeTransactionReasons = null, $WarehouseID = null) {
       
        if (!$FromDate || !$ToDate) {
            throw new Exception('getTransactions: To and from dates are required');
        }
        
        $data = [];
        
        if(!is_null($WarehouseID)) {        
            $data['WarehouseID'] = $WarehouseID  ;  
        }
        if(!is_null($PageNumber)) {
            $data['PageNumber'] = $PageNumber;
        }
        if(!is_null($PageSize)) {
            $data['PageSize'] = $PageSize;
        }
        $data['FromDate'] = $this->utils->formatDate($FromDate);
        $data['ToDate'] = $this->utils->formatDate($ToDate);
        if(!is_null($TransactionType) && is_array($TransactionType) && !empty($TransactionType)) {       
            $data['TransactionType'] = $TransactionType;
        }
        if(!is_null($TransactionReasons) && is_array($TransactionReasons) && !empty($TransactionReasons)) {         
            $data['TransactionReasons'] = $TransactionReasons;
        }
        if(!is_null($ExcludeTransactionReasons) && is_array($ExcludeTransactionReasons) && !empty($ExcludeTransactionReasons)) {         
            $data['ExcludeTransactionReasons'] = $ExcludeTransactionReasons;
        }
        $data['TenantToken'] = $this->tenantToken;
        $data['UserToken'] = $this->userToken;  
        return $this->remote->call($data, $this->pathPart .'/getTransactions');
    }
        
    /**
     * Get All Transactions History
     * @param datetime $FromDate - Max 1 week
     * @param datetime $ToDate
     * @param string $TransactionType - All, Add, Remove, or Pick
     * @param array $TransactionReasons
     * @param array $ExcludeTransactionReasons
     * @param int $WarehouseID
     * @return json object
     */
    public function getAllTransactions($FromDate, $ToDate, $TransactionType = null, $TransactionReasons = null, $ExcludeTransactionReasons = null, $WarehouseID = null) {
            
        $items = [];
        $pageSize = 10000;

        $pageNo = 0;
        while (1) {
            $transactions = $this->getTransactions($FromDate, $ToDate, $pageNo, $pageSize, $TransactionType, $TransactionReasons, $ExcludeTransactionReasons, $WarehouseID);
            $pageNo++;
            if (empty($transactions->Transactions)) {
                break;
            }
            $items = array_merge($items, $transactions->Transactions);
        }
        
        $result = (object)['Transactions'=>$items];
        
        return $result;
        
    }

}