<?php

/*
 * Copyright Talisman Innovations Ltd. (2016). All rights reserved.
 */

namespace SkuVault;

use SkuVault\Remote;
use SkuVault\Exception;
use SkuVault\Inventory;
use SkuVault\Products;
use SkuVault\PurchaseOrders;
use SkuVault\Sales;
use SkuVault\Utils;
use Psr\Log\LoggerInterface;

spl_autoload_register(function ($class) {
    list($namespace, $classname) = explode('\\', $class);
    if ($namespace == 'SkuVault') {
        include rtrim(__DIR__, '/').'/'.$classname . '.php';
    }
});

class Application {

    protected $username;
    protected $password;
    protected $tenantToken;
    protected $userToken;
    protected $remote;
    protected $sales;
    protected $inventory;
    protected $products;
    protected $order;
    protected $utils;

    public function __construct(LoggerInterface $logger, $username, $password) {
        $this->username = $username;
        $this->password = $password;
        $this->remote = new Remote($logger);
        $this->utils = new Utils();
        if($this->tenantToken && $this->userToken) {
            $this->sales = new Sales($this->tenantToken, $this->userToken);
            $this->products = new Products($this->tenantToken, $this->userToken);
            $this->order = new PurchaseOrders($this->tenantToken, $this->userToken);
            $this->inventory = new Inventory($this->tenantToken, $this->userToken);
        }
    }
    
    
    /**
     * Load sales class
     * @return class instance
     */
    public function sales() {
        
        if($this->tenantToken && $this->userToken) {
            return;
        }
        
        if(!$this->sales) {
            $this->sales = new Sales($this->tenantToken, $this->userToken);
        }
        
        return $this->sales;
    }
    
    /**
     * Load products class
     * @return class instance
     */
    public function products() {
        
        if($this->tenantToken && $this->userToken) {
            return;
        }
        
        if(!$this->products) {
            $this->products = new Products($this->tenantToken, $this->userToken);
        }
        
        return $this->products;
    }
    
    /**
     * Load order class
     * @return class instance
     */
    public function order() {
        
        if($this->tenantToken && $this->userToken) {
            return;
        }
        
        if(!$this->order) {
            $this->order = new PurchaseOrders($this->tenantToken, $this->userToken);
        }
        
        return $this->order;
    }

    /**
     * Load inventory class
     * @return class instance
     */
    public function inventory() {
        
        if($this->tenantToken && $this->userToken) {
            return;
        }
        
        if(!$this->inventory) {
            $this->inventory = new Inventory($this->tenantToken, $this->userToken);
        }
        
        return $this->inventory;
    }


    public function getTenantToken() {
        return $this->tenantToken;
    }

    public function getUserToken() {
        return $this->userToken;
    }
    
    /**
     * Get the Appliction tokens
     * @return json
     */
    public function getTokens() {

        $data = [
            'Email' => $this->username,
            'Password' => $this->password
        ];

        $result = $this->remote->call($data, 'getTokens');
        
        if(!isset($result->TenantToken) || is_null($result->TenantToken)) {
            throw new Exception('Null token values received please check your username and password');
        }

        $this->tenantToken = $result->TenantToken;
        $this->userToken = $result->UserToken;
        
        /**
         * TODO PERSIST THESE SOMEWHERE
         */
        return $result;
    }

}
