<?php

declare(strict_types=1);

namespace SalesforceRestApi;

use GuzzleHttp\Client;
use SalesforceRestApi\Client\SalesforceClient;
use SalesforceRestApi\Exceptions\ApiException;
use SalesforceRestApi\Models\ModelInterface;
use SalesforceRestApi\Resources\Query;
use SalesforceRestApi\Resources\Search;
use SalesforceRestApi\Resources\SObject;

class Salesforce
{
    private SalesforceClient $client;
    private ?Query $query = null;
    private ?Search $search = null;

    public function __construct(
        private string $instanceUrl,
        Client $httpClient,
        string $apiVersion = 'v65.0'
    ) {
        $this->client = new SalesforceClient($instanceUrl, $httpClient, $apiVersion);
    }

    /**
     * Get an SObject resource
     *
     * @param class-string<ModelInterface>|string $objectTypeOrClass SObject type string (e.g., "Account") or model class (e.g., Account::class)
     */
    public function sobject(string $objectTypeOrClass): SObject
    {
        return new SObject($this->client, $objectTypeOrClass);
    }

    /**
     * Get the Query resource
     */
    public function query(): Query
    {
        if ($this->query === null) {
            $this->query = new Query($this->client);
        }

        return $this->query;
    }

    /**
     * Get the Search resource
     */
    public function search(): Search
    {
        if ($this->search === null) {
            $this->search = new Search($this->client);
        }

        return $this->search;
    }

    /**
     * Get available API versions
     * @throws ApiException
     */
    public function versions(): array
    {
        $url = $this->instanceUrl . '/services/data';

        return $this->client->get($url);
    }

    /**
     * Get available resources for the current API version
     * @throws ApiException
     */
    public function resources(): array
    {
        return $this->client->get('/');
    }

    /**
     * Get all available SObjects
     * @throws ApiException
     */
    public function describeGlobal(): array
    {
        return $this->client->get('/sobjects');
    }
}
