<?php

declare(strict_types=1);

namespace SalesforceRestApi\Resources;

use SalesforceRestApi\Client\SalesforceClient;
use SalesforceRestApi\Models\ModelInterface;
use SalesforceRestApi\Models\SObjectRecord;

class SObject
{
    private readonly string $objectType;

    /**
     * @param class-string<ModelInterface>|string $objectTypeOrClass SObject type string (e.g., "Account") or model class (e.g., Account::class)
     */
    public function __construct(
        private readonly SalesforceClient $client,
        string $objectTypeOrClass
    ) {
        // Check if it's a class that implements ModelInterface
        if (class_exists($objectTypeOrClass) && is_subclass_of($objectTypeOrClass, ModelInterface::class)) {
            /** @var class-string<ModelInterface> $objectTypeOrClass */
            $this->objectType = $objectTypeOrClass::getSObjectType();
        } else {
            // It's a plain string like "Account"
            $this->objectType = $objectTypeOrClass;
        }
    }

    /**
     * Create a new record
     *
     * @template T of ModelInterface
     * @param class-string<T>|null $modelClass
     * @return T|array Response containing id and success status
     */
    public function createFromArray(array $data, ?string $modelClass = null): ModelInterface|array
    {
        return $this->client->post("/sobjects/{$this->objectType}", $data, $modelClass);
    }

    /**
     * Create a new record from a model object
     *
     */
    public function create(ModelInterface $model, ?string $modelClass = null): ModelInterface|array
    {
        $data = $model->toArray();
        return $this->client->post("/sobjects/{$this->objectType}", $data, $modelClass);
    }

    /**
     * Get a record by ID
     *
     * @template T of ModelInterface
     * @param class-string<T>|null $modelClass
     * @return T|SObjectRecord
     */
    public function get(string $id, ?string $modelClass = null, ?array $fields = null): ModelInterface|SObjectRecord
    {
        $queryParams = [];
        if ($fields !== null) {
            $queryParams['fields'] = implode(',', $fields);
        }

        $defaultModel = $modelClass ?? SObjectRecord::class;

        return $this->client->get("/sobjects/{$this->objectType}/{$id}", $defaultModel, $queryParams);
    }

    /**
     * Update a record by ID
     *
     * @return bool True if update was successful (204 No Content)
     */
    public function update(string $id, array $data): bool
    {
        $result = $this->client->patch("/sobjects/{$this->objectType}/{$id}", $data);

        return $result === null;
    }

    /**
     * Update a record from a model object
     *
     * @return bool True if update was successful (204 No Content)
     */
    public function updateFromModel(string $id, ModelInterface $model): bool
    {
        $data = $model->toArray();
        $result = $this->client->patch("/sobjects/{$this->objectType}/{$id}", $data);

        return $result === null;
    }

    /**
     * Delete a record by ID
     */
    public function delete(string $id): bool
    {
        return $this->client->delete("/sobjects/{$this->objectType}/{$id}");
    }

    /**
     * Upsert a record using external ID field
     *
     * @template T of ModelInterface
     * @param class-string<T>|null $modelClass
     * @return T|array
     */
    public function upsert(string $externalIdField, string $externalId, array $data, ?string $modelClass = null): ModelInterface|array
    {
        return $this->client->patch(
            "/sobjects/{$this->objectType}/{$externalIdField}/{$externalId}",
            $data,
            $modelClass
        );
    }

    /**
     * Get object metadata
     *
     * @template T of ModelInterface
     * @param class-string<T>|null $modelClass
     * @return T|array
     */
    public function describe(?string $modelClass = null): ModelInterface|array
    {
        return $this->client->get("/sobjects/{$this->objectType}/describe", $modelClass);
    }

    /**
     * Get basic object metadata
     *
     * @template T of ModelInterface
     * @param class-string<T>|null $modelClass
     * @return T|array
     */
    public function metadata(?string $modelClass = null): ModelInterface|array
    {
        return $this->client->get("/sobjects/{$this->objectType}", $modelClass);
    }

    /**
     * Get updated records within a date range
     *
     * @template T of ModelInterface
     * @param class-string<T>|null $modelClass
     * @return T|array
     */
    public function updated(string $startDate, string $endDate, ?string $modelClass = null): ModelInterface|array
    {
        return $this->client->get(
            "/sobjects/{$this->objectType}/updated",
            $modelClass,
            ['start' => $startDate, 'end' => $endDate]
        );
    }

    /**
     * Get deleted records within a date range
     *
     * @template T of ModelInterface
     * @param class-string<T>|null $modelClass
     * @return T|array
     */
    public function deleted(string $startDate, string $endDate, ?string $modelClass = null): ModelInterface|array
    {
        return $this->client->get(
            "/sobjects/{$this->objectType}/deleted",
            $modelClass,
            ['start' => $startDate, 'end' => $endDate]
        );
    }
}
