<?php

declare(strict_types=1);

namespace SalesforceRestApi\Resources;

use SalesforceRestApi\Client\SalesforceClient;
use SalesforceRestApi\Models\ModelInterface;
use SalesforceRestApi\Models\QueryResult;

class Query
{
    public function __construct(
        private readonly SalesforceClient $client
    ) {
    }

    /**
     * Execute a SOQL query
     *
     * @template T of ModelInterface
     * @param class-string<T>|null $modelClass
     * @return T|QueryResult
     */
    public function execute(string $soql, ?string $modelClass = null): ModelInterface|QueryResult
    {
        $defaultModel = $modelClass ?? QueryResult::class;

        return $this->client->get('/query', $defaultModel, ['q' => $soql]);
    }

    /**
     * Execute a SOQL query including deleted and archived records
     *
     * @template T of ModelInterface
     * @param class-string<T>|null $modelClass
     * @return T|QueryResult
     */
    public function queryAll(string $soql, ?string $modelClass = null): ModelInterface|QueryResult
    {
        $defaultModel = $modelClass ?? QueryResult::class;

        return $this->client->get('/queryAll', $defaultModel, ['q' => $soql]);
    }

    /**
     * Get next batch of records from a query
     *
     * @template T of ModelInterface
     * @param class-string<T>|null $modelClass
     * @return T|QueryResult
     */
    public function next(string $nextRecordsUrl, ?string $modelClass = null): ModelInterface|QueryResult
    {
        $defaultModel = $modelClass ?? QueryResult::class;

        // nextRecordsUrl comes from Salesforce and is already a full path
        return $this->client->get($nextRecordsUrl, $defaultModel);
    }

    /**
     * Execute a SOQL query and retrieve all records (handles pagination automatically)
     *
     * @template T of ModelInterface
     * @param class-string<T>|null $modelClass
     * @return array<T|QueryResult>
     */
    public function all(string $soql, ?string $modelClass = null): array
    {
        $defaultModel = $modelClass ?? QueryResult::class;
        $results = [];
        $result = $this->execute($soql, $defaultModel);

        $results[] = $result;

        while ($result instanceof QueryResult && !$result->isDone() && $result->getNextRecordsUrl()) {
            $result = $this->next($result->getNextRecordsUrl(), $defaultModel);
            $results[] = $result;
        }

        return $results;
    }
}
