<?php

declare(strict_types=1);

namespace SalesforceRestApi\Models;

use ArrayObject;
use Symfony\Component\Serializer\Exception\ExceptionInterface;
use Symfony\Component\Serializer\Normalizer\DenormalizerInterface;
use Symfony\Component\Serializer\Normalizer\NormalizerInterface;
use Symfony\Component\Serializer\Normalizer\ObjectNormalizer;

/**
 * Custom normalizer that handles Salesforce custom fields (ending in __c)
 * by separating them into the customData array
 */
class CustomFieldNormalizer implements NormalizerInterface, DenormalizerInterface
{
    private ObjectNormalizer $normalizer;

    public function __construct(ObjectNormalizer $normalizer)
    {
        $this->normalizer = $normalizer;
    }

    /**
     * Denormalizes data back into an object
     * Extracts custom fields (ending in __c) into customData array
     * @throws ExceptionInterface
     */
    public function denormalize(mixed $data, string $type, ?string $format = null, array $context = []): mixed
    {
        if (!is_array($data)) {
            return $this->normalizer->denormalize($data, $type, $format, $context);
        }

        // Separate custom fields from standard fields
        $customFields = [];
        $standardFields = [];

        foreach ($data as $key => $value) {
            // Check if the field name ends with __c (Salesforce custom field convention)
            if (is_string($key) && str_ends_with($key, '__c')) {
                $customFields[$key] = $value;
            } else {
                $standardFields[$key] = $value;
            }
        }

        // Denormalize the standard fields using the default normalizer
        $object = $this->normalizer->denormalize($standardFields, $type, $format, $context);

        // Set the custom fields into customData if the object is a BaseModel
        if ($object instanceof BaseModel && !empty($customFields)) {
            $object->setCustomData($customFields);
        }

        return $object;
    }

    /**
     * Checks whether the given class is supported for denormalization
     */
    public function supportsDenormalization(mixed $data, string $type, ?string $format = null, array $context = []): bool
    {
        return is_subclass_of($type, BaseModel::class);
    }

    /**
     * Normalizes an object into a set of arrays/scalars
     * Merges customData fields back into the main data array
     * @throws ExceptionInterface
     */
    public function normalize(mixed $data, ?string $format = null, array $context = []): array|string|int|float|bool|ArrayObject|null
    {
        // Get normalized data from the default normalizer
        $array = $this->normalizer->normalize($data, $format, $context);

        // If the object is a BaseModel, merge customData into the result
        if ($data instanceof BaseModel && is_array($array)) {
            $customData = $data->getCustomData();
            if (!empty($customData)) {
                $array = array_merge($array, $customData);
            }
        }

        return $array;
    }

    /**
     * Checks whether the given class is supported for normalization
     */
    public function supportsNormalization(mixed $data, ?string $format = null, array $context = []): bool
    {
        return $data instanceof BaseModel;
    }

    /**
     * Get supported types for denormalization
     */
    public function getSupportedTypes(?string $format): array
    {
        return [
            BaseModel::class => true,
        ];
    }
}
