<?php

declare(strict_types=1);

namespace SalesforceRestApi\Models;

use Symfony\Component\Serializer\Encoder\JsonEncoder;
use Symfony\Component\Serializer\Normalizer\ArrayDenormalizer;
use Symfony\Component\Serializer\Normalizer\ObjectNormalizer;
use Symfony\Component\Serializer\Serializer;
use Symfony\Component\Serializer\SerializerInterface;
use SalesforceRestApi\Models\CustomFieldNormalizer;

abstract class BaseModel implements ModelInterface
{
    protected array $rawData = [];
    protected array $customData = [];
    private static ?SerializerInterface $serializer = null;

    public function __construct(array $data = [])
    {
        $this->rawData = $data;
        $this->hydrate($data);
    }

    public static function fromArray(array $data): static
    {
        $serializer = self::getSerializer();

        // Denormalize array into the model object
        $instance = $serializer->denormalize($data, static::class, null, [
            ObjectNormalizer::DISABLE_TYPE_ENFORCEMENT => true,
        ]);

        // Store raw data
        $instance->rawData = $data;

        return $instance;
    }

    public function getRawData(): array
    {
        return $this->rawData;
    }

    public function toArray(): array
    {
        $serializer = self::getSerializer();

        // Normalize object to array, excluding internal arrays
        return $serializer->normalize($this, null, [
            ObjectNormalizer::SKIP_NULL_VALUES => true,
            ObjectNormalizer::IGNORED_ATTRIBUTES => ['rawData', 'customData'],
        ]);
    }

    /**
     * Override this method to perform additional hydration after Symfony Serializer
     */
    protected function hydrate(array $data): void
    {
        // Default implementation - can be overridden in child classes
        // This is now called after Symfony Serializer does its work
    }

    /**
     * Get a value from raw data with optional default
     */
    protected function get(string $key, mixed $default = null): mixed
    {
        return $this->rawData[$key] ?? $default;
    }

    /**
     * Set a value in raw data
     */
    protected function set(string $key, mixed $value): void
    {
        $this->rawData[$key] = $value;
    }

    /**
     * Get custom data (Salesforce custom fields ending in __c)
     */
    public function getCustomData(): array
    {
        return $this->customData;
    }

    /**
     * Set custom data
     */
    public function setCustomData(array $customData): void
    {
        $this->customData = $customData;
    }

    /**
     * Get a specific custom field value
     */
    public function getCustomField(string $fieldName): mixed
    {
        return $this->customData[$fieldName] ?? null;
    }

    /**
     * Set a specific custom field value
     */
    public function setCustomField(string $fieldName, mixed $value): void
    {
        $this->customData[$fieldName] = $value;
    }

    /**
     * Get or create the Symfony Serializer instance
     */
    protected static function getSerializer(): SerializerInterface
    {
        if (self::$serializer === null) {
            $objectNormalizer = new ObjectNormalizer();
            $normalizers = [
                new CustomFieldNormalizer($objectNormalizer),
                $objectNormalizer,
                new ArrayDenormalizer(),
            ];
            $encoders = [new JsonEncoder()];

            self::$serializer = new Serializer($normalizers, $encoders);
        }

        return self::$serializer;
    }

    /**
     * Set a custom serializer instance (useful for testing or custom configuration)
     */
    public static function setSerializer(SerializerInterface $serializer): void
    {
        self::$serializer = $serializer;
    }

    /**
     * Get the Salesforce SObject type name
     * By default, returns the short class name (e.g., "Account", "Contact")
     * Override this method if the SObject type differs from the class name
     */
    public static function getSObjectType(): string
    {
        $reflection = new \ReflectionClass(static::class);
        return $reflection->getShortName();
    }
}
