<?php

declare(strict_types=1);

namespace SalesforceRestApi\Client;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\RequestOptions;
use InvalidArgumentException;
use Psr\Http\Message\ResponseInterface;
use SalesforceRestApi\Exceptions\ApiException;
use SalesforceRestApi\Models\ModelInterface;

class SalesforceClient
{
    private const string API_VERSION = 'v65.0';
    private string $baseUrl;

    public function __construct(
        string                  $instanceUrl,
        private Client          $httpClient,
        private readonly string $apiVersion = self::API_VERSION
    )
    {
        $this->baseUrl = $instanceUrl . '/services/data/' . $this->apiVersion;
    }

    /**
     * Make a GET request
     *
     * @template T of ModelInterface
     * @param class-string<T>|null $modelClass
     * @return T|array
     * @throws ApiException
     */
    public function get(string $endpoint, ?string $modelClass = null, array $queryParams = []): ModelInterface|array
    {
        $response = $this->request('GET', $endpoint, [
            RequestOptions::QUERY => $queryParams,
        ]);

        return $this->parseResponse($response, $modelClass);
    }

    /**
     * Make a POST request
     *
     * @template T of ModelInterface
     * @param class-string<T>|null $modelClass
     * @return T|array
     * @throws ApiException
     */
    public function post(string $endpoint, array $data = [], ?string $modelClass = null): ModelInterface|array
    {
        $response = $this->request('POST', $endpoint, [
            RequestOptions::JSON => $data,
        ]);

        return $this->parseResponse($response, $modelClass);
    }

    /**
     * Make a PATCH request
     *
     * @template T of ModelInterface
     * @param class-string<T>|null $modelClass
     * @return T|array|null
     * @throws ApiException
     */
    public function patch(string $endpoint, array $data = [], ?string $modelClass = null): ModelInterface|array|null
    {
        $response = $this->request('PATCH', $endpoint, [
            RequestOptions::JSON => $data,
        ]);

        // PATCH for updates typically returns 204 No Content
        if ($response->getStatusCode() === 204) {
            return null;
        }

        return $this->parseResponse($response, $modelClass);
    }

    /**
     * Make a DELETE request
     * @throws ApiException
     */
    public function delete(string $endpoint): bool
    {
        $response = $this->request('DELETE', $endpoint);

        // DELETE typically returns 204 No Content on success
        return $response->getStatusCode() === 204;
    }

    /**
     * Get the base URL for API requests
     */
    public function getBaseUrl(): string
    {
        return $this->baseUrl;
    }

    /**
     * Get the API version
     */
    public function getApiVersion(): string
    {
        return $this->apiVersion;
    }

    /**
     * Make an HTTP request
     * @throws ApiException
     */
    private function request(string $method, string $endpoint, array $options = []): ResponseInterface
    {
        // Ensure endpoint starts with /
        if (!str_starts_with($endpoint, '/')) {
            $endpoint = '/' . $endpoint;
        }

        // If endpoint is a full URL, use it directly, otherwise prepend base URL
        $url = str_starts_with($endpoint, 'http') ? $endpoint : $this->baseUrl . $endpoint;

        try {
            return $this->httpClient->request($method, $url, $options);
        } catch (GuzzleException $e) {
            $response = method_exists($e, 'getResponse') ? $e->getResponse() : null;
            $errorData = null;

            if ($response) {
                $body = $response->getBody()->getContents();
                $errorData = json_decode($body, true);

                $message = sprintf(
                    'API request failed with status %d: %s',
                    $response->getStatusCode(),
                    $errorData[0]['message'] ?? $body
                );
            } else {
                $message = 'API request failed: ' . $e->getMessage();
            }

            throw new ApiException($message, $response, $errorData, previous: $e);
        }
    }

    /**
     * Parse response and optionally hydrate into model
     *
     * @template T of ModelInterface
     * @param class-string<T>|null $modelClass
     * @return T|array
     * @throws ApiException
     */
    private function parseResponse(ResponseInterface $response, ?string $modelClass = null): ModelInterface|array
    {
        $body = $response->getBody()->getContents();

        if (empty($body)) {
            return [];
        }

        $data = json_decode($body, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new ApiException('Failed to parse response JSON: ' . json_last_error_msg());
        }

        if ($modelClass === null) {
            return $data;
        }

        if (!is_subclass_of($modelClass, ModelInterface::class)) {
            throw new InvalidArgumentException(
                sprintf('Model class %s must implement %s', $modelClass, ModelInterface::class)
            );
        }

        return $modelClass::fromArray($data);
    }
}
