<?php

declare(strict_types=1);

require_once __DIR__ . '/../vendor/autoload.php';

use SalesforceRestApi\Models\BaseModel;
use SalesforceRestApi\Salesforce;
use Symfony\Component\Serializer\Attribute\SerializedName;

/**
 * Example 1: Using Symfony Serializer annotations for field mapping
 */
class Contact extends BaseModel
{
    #[SerializedName('Id')]
    public ?string $id = null;

    #[SerializedName('FirstName')]
    public ?string $firstName = null;

    #[SerializedName('LastName')]
    public ?string $lastName = null;

    #[SerializedName('Email')]
    public ?string $email = null;

    #[SerializedName('Phone')]
    public ?string $phone = null;

    #[SerializedName('CreatedDate')]
    public ?string $createdDate = null;

    public function getFullName(): string
    {
        return trim("{$this->firstName} {$this->lastName}");
    }

    public function isEmailValid(): bool
    {
        return $this->email !== null && filter_var($this->email, FILTER_VALIDATE_EMAIL) !== false;
    }
}

/**
 * Example 2: Advanced model with computed properties
 */
class Account extends BaseModel
{
    #[SerializedName('Id')]
    public ?string $id = null;

    #[SerializedName('Name')]
    public ?string $name = null;

    #[SerializedName('Industry')]
    public ?string $industry = null;

    #[SerializedName('AnnualRevenue')]
    public ?float $annualRevenue = null;

    #[SerializedName('NumberOfEmployees')]
    public ?int $numberOfEmployees = null;

    #[SerializedName('Website')]
    public ?string $website = null;

    #[SerializedName('BillingCity')]
    public ?string $billingCity = null;

    #[SerializedName('BillingState')]
    public ?string $billingState = null;

    /**
     * Additional hydration for custom logic
     */
    protected function hydrate(array $data): void
    {
        // You can still use hydrate() for custom processing after Symfony Serializer
        // For example, format or validate data
        if ($this->website && !str_starts_with($this->website, 'http')) {
            $this->website = 'https://' . $this->website;
        }
    }

    public function getRevenuePerEmployee(): ?float
    {
        if ($this->annualRevenue === null || $this->numberOfEmployees === null || $this->numberOfEmployees === 0) {
            return null;
        }

        return $this->annualRevenue / $this->numberOfEmployees;
    }

    public function isEnterprise(): bool
    {
        return $this->numberOfEmployees !== null && $this->numberOfEmployees >= 1000;
    }
}

/**
 * Example 3: Custom QueryResult with typed records
 */
class ContactQueryResult extends BaseModel
{
    public int $totalSize = 0;
    public bool $done = true;
    public ?string $nextRecordsUrl = null;

    /**
     * @var array<int, Contact>
     */
    public array $records = [];

    protected function hydrate(array $data): void
    {
        if (isset($data['records']) && is_array($data['records'])) {
            $this->records = array_map(
                fn(array $record) => Contact::fromArray($record),
                $data['records']
            );
        }
    }

    /**
     * @return array<int, Contact>
     */
    public function getContacts(): array
    {
        return $this->records;
    }

    public function filterByValidEmail(): array
    {
        return array_filter($this->records, fn(Contact $c) => $c->isEmailValid());
    }
}

// Initialize Salesforce
$salesforce = Salesforce::withPassword(
    clientId: 'your_client_id',
    clientSecret: 'your_client_secret',
    username: 'your_username@example.com',
    password: 'your_password'
);

echo "=== Symfony Serializer Integration Examples ===\n\n";

// Example 1: Fetch a contact with automatic hydration
echo "1. Fetching contact with Symfony Serializer:\n";
$contact = $salesforce->sobject('Contact')->get('003xx000004TmiXXXX', Contact::class);

echo "   Full Name: {$contact->getFullName()}\n";
echo "   Email: {$contact->email}\n";
echo "   Email Valid: " . ($contact->isEmailValid() ? 'Yes' : 'No') . "\n";
echo "   Created: {$contact->createdDate}\n\n";

// Example 2: Query contacts with custom model
echo "2. Querying contacts with custom model:\n";
$result = $salesforce->query()->execute(
    "SELECT Id, FirstName, LastName, Email, Phone, CreatedDate FROM Contact LIMIT 5",
    ContactQueryResult::class
);

echo "   Found {$result->totalSize} contacts:\n";
foreach ($result->getContacts() as $c) {
    echo "   - {$c->getFullName()} ({$c->email})\n";
}

$validContacts = $result->filterByValidEmail();
echo "   Valid emails: " . count($validContacts) . "\n\n";

// Example 3: Fetch account with computed properties
echo "3. Fetching account with computed properties:\n";
$account = $salesforce->sobject('Account')->get('001xx000003DGbXXXX', Account::class);

echo "   Name: {$account->name}\n";
echo "   Industry: {$account->industry}\n";
echo "   Employees: {$account->numberOfEmployees}\n";
echo "   Annual Revenue: $" . number_format($account->annualRevenue ?? 0) . "\n";
echo "   Revenue/Employee: $" . number_format($account->getRevenuePerEmployee() ?? 0) . "\n";
echo "   Enterprise: " . ($account->isEnterprise() ? 'Yes' : 'No') . "\n";
echo "   Website: {$account->website}\n\n";

// Example 4: Serializing back to array
echo "4. Serializing model back to array:\n";
$contactArray = $contact->toArray();
print_r($contactArray);
echo "\n";

// Example 5: Creating a contact and hydrating response
echo "5. Creating contact with automatic response hydration:\n";
$newContactData = [
    'FirstName' => 'Jane',
    'LastName' => 'Doe',
    'Email' => 'jane.doe@example.com',
    'Phone' => '555-9999',
];

$createResult = $salesforce->sobject('Contact')->createFromArray($newContactData);
echo "   Created contact ID: {$createResult['id']}\n";

// Fetch the newly created contact
$newContact = $salesforce->sobject('Contact')->get($createResult['id'], Contact::class);
echo "   Full Name: {$newContact->getFullName()}\n";
echo "   Email Valid: " . ($newContact->isEmailValid() ? 'Yes' : 'No') . "\n\n";

echo "=== All examples completed successfully! ===\n";
