<?php

declare(strict_types=1);

require_once __DIR__ . '/../vendor/autoload.php';

use SalesforceRestApi\Models\Standard\Account;
use SalesforceRestApi\Models\Standard\Contact;
use SalesforceRestApi\Models\Standard\Lead;
use SalesforceRestApi\Models\Standard\Opportunity;
use SalesforceRestApi\Models\Standard\Task;
use SalesforceRestApi\Models\Standard\Campaign;
use SalesforceRestApi\Salesforce;

// Initialize Salesforce
$salesforce = Salesforce::withPassword(
    clientId: 'your_client_id',
    clientSecret: 'your_client_secret',
    username: 'your_username@example.com',
    password: 'your_password'
);

echo "=== Salesforce Standard Models Demo ===\n\n";

// Example 1: Working with Accounts
echo "1. Account Operations:\n";
$account = $salesforce->sobject(Account::class)->get('001xx000003DGbXXXX', Account::class);

echo "   Company: {$account->name}\n";
echo "   Industry: {$account->industry}\n";
echo "   Employees: {$account->numberOfEmployees}\n";
echo "   Annual Revenue: $" . number_format($account->annualRevenue ?? 0) . "\n";
echo "   Revenue/Employee: $" . number_format($account->getRevenuePerEmployee() ?? 0, 2) . "\n";
echo "   Enterprise: " . ($account->isEnterprise() ? 'Yes' : 'No') . "\n\n";

// Example 2: Working with Contacts
echo "2. Contact Operations:\n";
$contact = $salesforce->sobject(Contact::class)->get('003xx000004TmiXXXX', Contact::class);

echo "   Full Name: {$contact->getFullName()}\n";
echo "   Title: {$contact->title}\n";
echo "   Email: {$contact->email}\n";
echo "   Email Valid: " . ($contact->isEmailValid() ? 'Yes' : 'No') . "\n";
echo "   Can Contact: " . ($contact->canContact() ? 'Yes' : 'No') . "\n";
echo "   Mailing Address: {$contact->getMailingAddress()}\n\n";

// Example 3: Working with Leads
echo "3. Lead Operations:\n";
$lead = $salesforce->sobject(Lead::class)->get('00Qxx000001234XXX', Lead::class);

echo "   Name: {$lead->getFullName()}\n";
echo "   Company: {$lead->company}\n";
echo "   Status: {$lead->status}\n";
echo "   Qualified: " . ($lead->isQualified() ? 'Yes' : 'No') . "\n";
echo "   Converted: " . ($lead->isConverted ? 'Yes' : 'No') . "\n";
echo "   Email Valid: " . ($lead->isEmailValid() ? 'Yes' : 'No') . "\n\n";

// Example 4: Working with Opportunities
echo "4. Opportunity Operations:\n";
$opp = $salesforce->sobject(Opportunity::class)->get('006xx000001234XXX', Opportunity::class);

echo "   Name: {$opp->name}\n";
echo "   Stage: {$opp->stageName}\n";
echo "   Amount: $" . number_format($opp->amount ?? 0) . "\n";
echo "   Probability: {$opp->probability}%\n";
echo "   Expected Revenue: $" . number_format($opp->getExpectedRevenue()) . "\n";
echo "   Close Date: {$opp->closeDate}\n";
echo "   Days Until Close: {$opp->getDaysUntilClose()}\n";
echo "   Closing Soon: " . ($opp->isClosingSoon(30) ? 'Yes' : 'No') . "\n";
echo "   High Value: " . ($opp->isHighValue(100000) ? 'Yes' : 'No') . "\n\n";

// Example 5: Working with Tasks
echo "5. Task Operations:\n";
$task = $salesforce->sobject(Task::class)->get('00Txx000001234XXX', Task::class);

echo "   Subject: {$task->subject}\n";
echo "   Status: {$task->status}\n";
echo "   Priority: {$task->priority}\n";
echo "   Due Date: {$task->activityDate}\n";
echo "   Days Until Due: {$task->getDaysUntilDue()}\n";
echo "   Overdue: " . ($task->isOverdue() ? 'Yes' : 'No') . "\n";
echo "   Closed: " . ($task->isClosed() ? 'Yes' : 'No') . "\n\n";

// Example 6: Working with Campaigns
echo "6. Campaign Operations:\n";
$campaign = $salesforce->sobject(Campaign::class)->get('701xx000001234XXX', Campaign::class);

echo "   Name: {$campaign->name}\n";
echo "   Status: {$campaign->status}\n";
echo "   Budgeted Cost: $" . number_format($campaign->budgetedCost ?? 0) . "\n";
echo "   Actual Cost: $" . number_format($campaign->actualCost ?? 0) . "\n";
echo "   ROI: " . number_format($campaign->getROI() ?? 0, 2) . "%\n";
echo "   Conversion Rate: " . number_format($campaign->getConversionRate() ?? 0, 2) . "%\n";
echo "   Response Rate: " . number_format($campaign->getResponseRate() ?? 0, 2) . "%\n";
echo "   Active: " . ($campaign->isActive ? 'Yes' : 'No') . "\n";
echo "   Ongoing: " . ($campaign->isOngoing() ? 'Yes' : 'No') . "\n\n";

// Example 7: Querying with Standard Models
echo "7. Querying Accounts:\n";
$result = $salesforce->query()->execute(
    "SELECT Id, Name, Industry, AnnualRevenue, NumberOfEmployees
     FROM Account
     WHERE Industry = 'Technology'
     LIMIT 5",
    Account::class
);

foreach ($result->getRecords() as $acc) {
    echo "   - {$acc->name} ({$acc->industry}) - " .
         ($acc->isEnterprise() ? 'Enterprise' : 'SMB') . "\n";
}
echo "\n";

// Example 8: Finding Overdue Tasks
echo "8. Finding Overdue Tasks:\n";
$taskResult = $salesforce->query()->execute(
    "SELECT Id, Subject, ActivityDate, Status, Priority
     FROM Task
     WHERE Status != 'Completed'
     AND ActivityDate < TODAY
     LIMIT 5",
    Task::class
);

foreach ($taskResult->getRecords() as $t) {
    if ($t->isOverdue()) {
        echo "   - {$t->subject} (Overdue by " . abs($t->getDaysUntilDue()) . " days)\n";
    }
}
echo "\n";

// Example 9: Opportunities Closing Soon
echo "9. Opportunities Closing Soon:\n";
$oppResult = $salesforce->query()->execute(
    "SELECT Id, Name, Amount, CloseDate, StageName, Probability
     FROM Opportunity
     WHERE IsClosed = false
     AND CloseDate <= NEXT_N_DAYS:30
     LIMIT 5",
    Opportunity::class
);

foreach ($oppResult->getRecords() as $o) {
    if ($o->isClosingSoon(30)) {
        echo "   - {$o->name}: Closes in {$o->getDaysUntilClose()} days " .
             "(\${$o->amount}, {$o->probability}%)\n";
    }
}
echo "\n";

// Example 10: Creating Records with Standard Models
echo "10. Creating New Contact:\n";
$newContactData = [
    'FirstName' => 'John',
    'LastName' => 'Doe',
    'Email' => 'john.doe@example.com',
    'Phone' => '555-1234',
    'Title' => 'VP of Sales',
    'AccountId' => '001xx000003DGbXXXX',
];

$createResult = $salesforce->sobject(Contact::class)->createFromArray($newContactData);
echo "   Created Contact ID: {$createResult['id']}\n";

// Fetch the newly created contact
$newContact = $salesforce->sobject(Contact::class)->get($createResult['id'], Contact::class);
echo "   Full Name: {$newContact->getFullName()}\n";
echo "   Email Valid: " . ($newContact->isEmailValid() ? 'Yes' : 'No') . "\n\n";

echo "=== Demo Complete ===\n";
