<?php

declare(strict_types=1);

require_once __DIR__ . '/../vendor/autoload.php';

use SalesforceRestApi\Models\Standard\Account;
use SalesforceRestApi\Models\Standard\Contact;
use SalesforceRestApi\Models\Standard\Order;
use SalesforceRestApi\Models\Standard\OrderItem;
use SalesforceRestApi\Salesforce;

/**
 * Example: CRUD Operations with Model Objects
 *
 * This example demonstrates using createFromModel() and updateFromModel()
 * methods to work directly with model objects instead of arrays.
 */

// Initialize Salesforce
$salesforce = Salesforce::withPassword(
    clientId: 'your_client_id',
    clientSecret: 'your_client_secret',
    username: 'your_username@example.com',
    password: 'your_password'
);

// ============================================================================
// Example 1a: Create Account using Array (Traditional Method)
// ============================================================================
echo "=== Example 1a: Creating Account from Array ===\n\n";

// Traditional approach: create using an array
$accountData = [
    'Name' => 'Array-Based Corp',
    'Industry' => 'Technology',
    'Phone' => '555-1111',
    'Website' => 'https://arraybased.example',
    'AnnualRevenue' => 3000000.00,
    'NumberOfEmployees' => 100,
    'BillingStreet' => '456 Array Ave',
    'BillingCity' => 'San Francisco',
    'BillingState' => 'CA',
    'BillingPostalCode' => '94103',
    'BillingCountry' => 'USA',
    // Custom fields
    'CustomerTier__c' => 'Silver',
    'LifetimeValue__c' => 150000.00,
    'Active__c' => true,
];

// Use createFromArray() for array-based creation
$arrayResult = $salesforce->sobject(Account::class)->createFromArray($accountData);

echo "Created Account (from array):\n";
echo "  ID: {$arrayResult['id']}\n";
echo "  Success: " . ($arrayResult['success'] ? 'Yes' : 'No') . "\n\n";

// ============================================================================
// Example 1b: Create Account using Model Object (Recommended)
// ============================================================================
echo "=== Example 1b: Creating Account from Model (Recommended) ===\n\n";

// Modern approach: create using a model object
$account = new Account();
$account->name = 'Tech Innovations Inc';
$account->industry = 'Technology';
$account->phone = '555-1234';
$account->website = 'https://techinnovations.example';
$account->annualRevenue = 5000000.00;
$account->numberOfEmployees = 150;
$account->billingStreet = '123 Tech Street';
$account->billingCity = 'San Francisco';
$account->billingState = 'CA';
$account->billingPostalCode = '94102';
$account->billingCountry = 'USA';

// Add custom fields using customData
$account->setCustomField('CustomerTier__c', 'Gold');
$account->setCustomField('LifetimeValue__c', 250000.00);
$account->setCustomField('Active__c', true);

// Use create() for model-based creation
$result = $salesforce->sobject(Account::class)->create($account);

echo "Created Account (from model):\n";
echo "  ID: {$result['id']}\n";
echo "  Success: " . ($result['success'] ? 'Yes' : 'No') . "\n\n";

echo "Model approach benefits:\n";
echo "  ✓ Type-safe with IDE autocomplete\n";
echo "  ✓ Can add business logic and validation\n";
echo "  ✓ Cleaner, more maintainable code\n";
echo "  ✓ Automatic custom field handling\n\n";

$accountId = $result['id'];

// ============================================================================
// Example 2: Retrieve and Update Account
// ============================================================================
echo "=== Example 2: Retrieving and Updating Account ===\n\n";

// Retrieve the account as a model object
$retrievedAccount = $salesforce->sobject(Account::class)->get($accountId, Account::class);

echo "Retrieved Account:\n";
echo "  Name: {$retrievedAccount->name}\n";
echo "  Industry: {$retrievedAccount->industry}\n";
echo "  Annual Revenue: \${$retrievedAccount->annualRevenue}\n";
echo "  Customer Tier: {$retrievedAccount->getCustomField('CustomerTier__c')}\n\n";

// Modify the account
$retrievedAccount->phone = '555-5678';
$retrievedAccount->annualRevenue = 7500000.00;
$retrievedAccount->setCustomField('CustomerTier__c', 'Platinum');
$retrievedAccount->setCustomField('LifetimeValue__c', 500000.00);

// Update using the model object (can also use update() with array)
$updateSuccess = $salesforce->sobject(Account::class)->updateFromModel($accountId, $retrievedAccount);

echo "Update Result: " . ($updateSuccess ? 'Success' : 'Failed') . "\n\n";

// ============================================================================
// Example 3: Create Contact with Account Relationship
// ============================================================================
echo "=== Example 3: Creating Contact from Model ===\n\n";

$contact = new Contact();
$contact->accountId = $accountId;
$contact->firstName = 'Jane';
$contact->lastName = 'Smith';
$contact->email = 'jane.smith@techinnovations.example';
$contact->phone = '555-9999';
$contact->title = 'CTO';
$contact->department = 'Engineering';
$contact->mailingStreet = '123 Tech Street';
$contact->mailingCity = 'San Francisco';
$contact->mailingState = 'CA';
$contact->mailingPostalCode = '94102';
$contact->mailingCountry = 'USA';

// Add custom fields
$contact->setCustomField('LoyaltyPoints__c', 1000);
$contact->setCustomField('PreferredContactMethod__c', 'Email');

$contactResult = $salesforce->sobject(Contact::class)->create($contact);

echo "Created Contact:\n";
echo "  ID: {$contactResult['id']}\n";
echo "  Name: {$contact->firstName} {$contact->lastName}\n";
echo "  Account ID: {$contact->accountId}\n\n";

$contactId = $contactResult['id'];

// ============================================================================
// Example 4: Create Order from Model
// ============================================================================
echo "=== Example 4: Creating Order from Model ===\n\n";

$order = new Order();
$order->accountId = $accountId;
$order->effectiveDate = date('Y-m-d');
$order->status = 'Draft';
$order->billingStreet = '123 Tech Street';
$order->billingCity = 'San Francisco';
$order->billingState = 'CA';
$order->billingPostalCode = '94102';
$order->billingCountry = 'USA';
$order->shippingStreet = '123 Tech Street';
$order->shippingCity = 'San Francisco';
$order->shippingState = 'CA';
$order->shippingPostalCode = '94102';
$order->shippingCountry = 'USA';

// Add custom fields
$order->setCustomField('ShippingMethod__c', 'Express');
$order->setCustomField('PriorityOrder__c', true);
$order->setCustomField('DiscountCode__c', 'WELCOME20');

$orderResult = $salesforce->sobject(Order::class)->create($order);

echo "Created Order:\n";
echo "  ID: {$orderResult['id']}\n";
echo "  Status: {$order->status}\n";
echo "  Effective Date: {$order->effectiveDate}\n\n";

$orderId = $orderResult['id'];

// ============================================================================
// Example 5: Bulk Create using Models
// ============================================================================
echo "=== Example 5: Bulk Create Multiple Contacts ===\n\n";

$contacts = [];
$contactNames = [
    ['John', 'Doe', 'john.doe@example.com'],
    ['Alice', 'Johnson', 'alice.j@example.com'],
    ['Bob', 'Williams', 'bob.w@example.com'],
];

foreach ($contactNames as [$firstName, $lastName, $email]) {
    $newContact = new Contact();
    $newContact->accountId = $accountId;
    $newContact->firstName = $firstName;
    $newContact->lastName = $lastName;
    $newContact->email = $email;
    $newContact->setCustomField('LoyaltyPoints__c', rand(100, 1000));

    $result = $salesforce->sobject(Contact::class)->create($newContact);
    $contacts[] = $result['id'];

    echo "Created: {$firstName} {$lastName} (ID: {$result['id']})\n";
}

echo "\nTotal contacts created: " . count($contacts) . "\n\n";

// ============================================================================
// Example 6: Complex Update with Custom Fields
// ============================================================================
echo "=== Example 6: Complex Update Operation ===\n\n";

// Retrieve the account again
$accountToUpdate = $salesforce->sobject(Account::class)->get($accountId, Account::class);

// Update multiple fields including custom ones
$accountToUpdate->annualRevenue = 10000000.00;
$accountToUpdate->numberOfEmployees = 250;
$accountToUpdate->rating = 'Hot';
$accountToUpdate->setCustomField('CustomerTier__c', 'Diamond');
$accountToUpdate->setCustomField('LifetimeValue__c', 1000000.00);
$accountToUpdate->setCustomField('PreferredPaymentMethod__c', 'Wire Transfer');

// Perform update
$success = $salesforce->sobject(Account::class)->updateFromModel($accountId, $accountToUpdate);

echo "Complex Update Result: " . ($success ? 'Success' : 'Failed') . "\n";

// Verify the update
$verifyAccount = $salesforce->sobject(Account::class)->get($accountId, Account::class);
echo "\nVerified Updates:\n";
echo "  Annual Revenue: \${$verifyAccount->annualRevenue}\n";
echo "  Employees: {$verifyAccount->numberOfEmployees}\n";
echo "  Rating: {$verifyAccount->rating}\n";
echo "  Customer Tier: {$verifyAccount->getCustomField('CustomerTier__c')}\n";
echo "  Lifetime Value: \${$verifyAccount->getCustomField('LifetimeValue__c')}\n\n";

// ============================================================================
// Example 7: Working with Relationships
// ============================================================================
echo "=== Example 7: Query and Update Related Records ===\n\n";

// Query all contacts for this account
$contactsResult = $salesforce->query()->execute(
    "SELECT Id, FirstName, LastName, Email, LoyaltyPoints__c
     FROM Contact
     WHERE AccountId = '{$accountId}'",
    Contact::class
);

echo "Found {$contactsResult->totalSize} contacts:\n\n";

foreach ($contactsResult->records as $contactModel) {
    echo "- {$contactModel->firstName} {$contactModel->lastName}\n";
    echo "  Email: {$contactModel->email}\n";
    echo "  Loyalty Points: {$contactModel->getCustomField('LoyaltyPoints__c')}\n";

    // Give bonus points
    $currentPoints = $contactModel->getCustomField('LoyaltyPoints__c') ?? 0;
    $contactModel->setCustomField('LoyaltyPoints__c', $currentPoints + 100);

    // Update the contact
    $salesforce->sobject(Contact::class)->updateFromModel($contactModel->id, $contactModel);
    echo "  Updated to: " . ($currentPoints + 100) . " points\n\n";
}

// ============================================================================
// Summary
// ============================================================================
echo "=== Summary ===\n\n";
echo "Benefits of createFromModel() and updateFromModel():\n";
echo "1. Type-safe operations with IDE autocomplete\n";
echo "2. Automatic handling of custom fields via customData\n";
echo "3. Model validation and business logic\n";
echo "4. Cleaner, more maintainable code\n";
echo "5. Consistent API with the rest of the library\n";
echo "6. No manual array construction needed\n\n";

echo "Comparison:\n\n";

echo "Old way (array-based):\n";
echo "  \$data = [\n";
echo "    'Name' => 'Test',\n";
echo "    'Industry' => 'Tech',\n";
echo "    'CustomField__c' => 'value'\n";
echo "  ];\n";
echo "  \$result = \$salesforce->sobject(Account::class)->create(\$data);\n\n";

echo "New way (model-based):\n";
echo "  \$account = new Account();\n";
echo "  \$account->name = 'Test';\n";
echo "  \$account->industry = 'Tech';\n";
echo "  \$account->setCustomField('CustomField__c', 'value');\n";
echo "  \$result = \$salesforce->sobject(Account::class)->create(\$account);\n";
