<?php

declare(strict_types=1);

require_once __DIR__ . '/../vendor/autoload.php';

use SalesforceRestApi\Models\BaseModel;
use SalesforceRestApi\Salesforce;
use Symfony\Component\Serializer\Attribute\SerializedName;

// Define a custom Account model
// With Symfony Serializer, you just need to define properties with SerializedName attributes
// No need to manually map fields in hydrate() anymore!
class Account extends BaseModel
{
    #[SerializedName('Id')]
    public ?string $id = null;

    #[SerializedName('Name')]
    public ?string $name = null;

    #[SerializedName('Industry')]
    public ?string $industry = null;

    #[SerializedName('Phone')]
    public ?string $phone = null;

    #[SerializedName('Website')]
    public ?string $website = null;

    // You can still use hydrate() for custom logic if needed
    protected function hydrate(array $data): void
    {
        // Example: Normalize website URL
        if ($this->website && !str_starts_with($this->website, 'http')) {
            $this->website = 'https://' . $this->website;
        }
    }

    public function getId(): ?string
    {
        return $this->id;
    }

    public function getName(): ?string
    {
        return $this->name;
    }

    public function getIndustry(): ?string
    {
        return $this->industry;
    }

    public function getPhone(): ?string
    {
        return $this->phone;
    }

    public function getWebsite(): ?string
    {
        return $this->website;
    }
}

// Define a custom QueryResult model
// Symfony Serializer handles primitive types automatically
class AccountQueryResult extends BaseModel
{
    public int $totalSize = 0;
    public bool $done = true;

    /**
     * @var array<int, Account>
     */
    public array $records = [];

    // Only need hydrate() for converting nested arrays to model instances
    protected function hydrate(array $data): void
    {
        if (isset($data['records']) && is_array($data['records'])) {
            $this->records = array_map(
                fn(array $record) => Account::fromArray($record),
                $data['records']
            );
        }
    }

    public function getAccounts(): array
    {
        return $this->records;
    }
}

// Initialize Salesforce
$salesforce = Salesforce::withPassword(
    clientId: 'your_client_id',
    clientSecret: 'your_client_secret',
    username: 'your_username@example.com',
    password: 'your_password'
);

// Fetch an account using custom model
// Pass the model class to automatically derive the SObject type
$accountApi = $salesforce->sobject(Account::class);
$account = $accountApi->get('001xx000003DGbXXXX', Account::class);

echo "Account Name: {$account->getName()}\n";
echo "Industry: {$account->getIndustry()}\n";
echo "Phone: {$account->getPhone()}\n";

// Query with custom model
$result = $salesforce->query()->execute(
    "SELECT Id, Name, Industry, Phone, Website FROM Account LIMIT 10",
    AccountQueryResult::class
);

echo "\nFound {$result->totalSize} accounts:\n";

foreach ($result->getAccounts() as $acc) {
    echo "- {$acc->getName()} ({$acc->getIndustry()})\n";
}
