<?php

declare(strict_types=1);

require_once __DIR__ . '/../vendor/autoload.php';

use SalesforceRestApi\Models\Standard\Account;
use SalesforceRestApi\Models\Standard\Contact;
use SalesforceRestApi\Models\Standard\Order;
use SalesforceRestApi\Salesforce;

/**
 * Example: Using customData for Salesforce Custom Fields
 *
 * The BaseModel now automatically captures all Salesforce custom fields (ending in __c)
 * into the customData array. This means you don't need to explicitly define custom fields
 * in your models unless you want typed properties and IDE support.
 *
 * This approach is useful when:
 * - You have many custom fields and don't want to define them all
 * - Custom fields vary between Salesforce orgs
 * - You want dynamic handling of custom fields
 */

// Initialize Salesforce
$salesforce = Salesforce::withPassword(
    clientId: 'your_client_id',
    clientSecret: 'your_client_secret',
    username: 'your_username@example.com',
    password: 'your_password'
);

// Example 1: Query Account with Custom Fields - No Model Extension Needed
echo "=== Example 1: Account with Custom Fields (No Extension) ===\n\n";

$accountResult = $salesforce->query()->execute(
    "SELECT Id, Name, Industry, Phone,
            CustomerTier__c, LifetimeValue__c, CustomerSince__c,
            Active__c, PreferredPaymentMethod__c
     FROM Account
     WHERE Active__c = true
     LIMIT 1"
);

if (!empty($accountResult->records)) {
    $accountData = $accountResult->records[0];
    $account = Account::fromArray($accountData);

    echo "Account: {$account->name}\n";
    echo "Industry: {$account->industry}\n";
    echo "Phone: {$account->phone}\n";
    echo "\n";

    // Access custom fields via customData
    echo "Custom Fields:\n";
    foreach ($account->getCustomData() as $fieldName => $value) {
        echo "  {$fieldName}: {$value}\n";
    }
    echo "\n";

    // Access specific custom field
    $customerTier = $account->getCustomField('CustomerTier__c');
    echo "Customer Tier: {$customerTier}\n";

    $lifetimeValue = $account->getCustomField('LifetimeValue__c');
    echo "Lifetime Value: \${$lifetimeValue}\n";

    $isActive = $account->getCustomField('Active__c');
    echo "Active: " . ($isActive ? 'Yes' : 'No') . "\n";
    echo "\n";
}

// Example 2: Contact with Custom Fields
echo "=== Example 2: Contact with Custom Fields ===\n\n";

$contactResult = $salesforce->query()->execute(
    "SELECT Id, FirstName, LastName, Email,
            LoyaltyPoints__c, PreferredContactMethod__c,
            TwitterHandle__c, LastPurchaseDate__c
     FROM Contact
     WHERE LoyaltyPoints__c > 0
     LIMIT 1"
);

if (!empty($contactResult->records)) {
    $contactData = $contactResult->records[0];
    $contact = Contact::fromArray($contactData);

    echo "Contact: {$contact->firstName} {$contact->lastName}\n";
    echo "Email: {$contact->email}\n";
    echo "\n";

    echo "Custom Fields:\n";
    $customData = $contact->getCustomData();
    echo "  Loyalty Points: {$customData['LoyaltyPoints__c']}\n";
    echo "  Preferred Contact: {$customData['PreferredContactMethod__c']}\n";
    echo "  Twitter: {$customData['TwitterHandle__c']}\n";
    echo "  Last Purchase: {$customData['LastPurchaseDate__c']}\n";
    echo "\n";
}

// Example 3: Creating Record with Custom Fields
echo "=== Example 3: Creating Record with Custom Fields ===\n\n";

// Create an Account model with standard fields
$newAccount = new Account();
$newAccount->name = 'Dynamic Custom Fields Corp';
$newAccount->industry = 'Technology';
$newAccount->phone = '555-9999';

// Add custom fields to customData
$newAccount->setCustomField('CustomerTier__c', 'Gold');
$newAccount->setCustomField('LifetimeValue__c', 150000.00);
$newAccount->setCustomField('CustomerSince__c', '2025-01-01');
$newAccount->setCustomField('Active__c', true);
$newAccount->setCustomField('PreferredPaymentMethod__c', 'Wire Transfer');

// Convert to array for API call
$accountData = $newAccount->toArray();

echo "Data to send to Salesforce:\n";
print_r($accountData);
echo "\n";

// Create in Salesforce
$result = $salesforce->sobject('Account')->createFromArray($accountData);
echo "Created Account ID: {$result['id']}\n\n";

// Example 4: Updating Custom Fields
echo "=== Example 4: Updating Custom Fields ===\n\n";

// Retrieve the account
$retrievedAccount = $salesforce->sobject('Account')->get($result['id'], Account::class);

echo "Current Custom Data:\n";
print_r($retrievedAccount->getCustomData());
echo "\n";

// Update custom fields
$retrievedAccount->setCustomField('CustomerTier__c', 'Platinum');
$retrievedAccount->setCustomField('LifetimeValue__c', 300000.00);

// Update in Salesforce
$updateData = $retrievedAccount->toArray();
$salesforce->sobject('Account')->update($retrievedAccount->id, $updateData);

echo "Updated Account to Platinum tier\n\n";

// Example 5: Dynamic Custom Field Handling
echo "=== Example 5: Dynamic Custom Field Handling ===\n\n";

// Simulate receiving data with unknown custom fields
$dynamicData = [
    'Id' => '001xx000003DGbXXXX',
    'Name' => 'Dynamic Corp',
    'Industry' => 'Finance',
    // These custom fields weren't predefined in the model
    'CustomField1__c' => 'Value 1',
    'CustomField2__c' => 42,
    'CustomField3__c' => true,
    'AnotherCustomField__c' => ['nested' => 'data'],
];

$dynamicAccount = Account::fromArray($dynamicData);

echo "Standard Fields:\n";
echo "  ID: {$dynamicAccount->id}\n";
echo "  Name: {$dynamicAccount->name}\n";
echo "  Industry: {$dynamicAccount->industry}\n";
echo "\n";

echo "All Custom Fields (automatically captured):\n";
foreach ($dynamicAccount->getCustomData() as $field => $value) {
    $valueStr = is_array($value) ? json_encode($value) : (string)$value;
    echo "  {$field}: {$valueStr}\n";
}
echo "\n";

// Example 6: Order with Custom Fields
echo "=== Example 6: Order with Custom Fields ===\n\n";

$orderResult = $salesforce->query()->execute(
    "SELECT Id, OrderNumber, Status, TotalAmount,
            ShippingMethod__c, PriorityOrder__c,
            DiscountCode__c, CommissionRate__c
     FROM Order
     WHERE Status = 'Activated'
     LIMIT 1"
);

if (!empty($orderResult->records)) {
    $orderData = $orderResult->records[0];
    $order = Order::fromArray($orderData);

    echo "Order: {$order->orderNumber}\n";
    echo "Status: {$order->status}\n";
    echo "Total: \${$order->totalAmount}\n";
    echo "\n";

    echo "Custom Order Fields:\n";
    $shippingMethod = $order->getCustomField('ShippingMethod__c');
    $isPriority = $order->getCustomField('PriorityOrder__c');
    $discountCode = $order->getCustomField('DiscountCode__c');
    $commissionRate = $order->getCustomField('CommissionRate__c');

    echo "  Shipping Method: {$shippingMethod}\n";
    echo "  Priority Order: " . ($isPriority ? 'Yes' : 'No') . "\n";
    echo "  Discount Code: {$discountCode}\n";
    echo "  Commission Rate: {$commissionRate}%\n";

    if ($commissionRate && $order->totalAmount) {
        $commission = $order->totalAmount * ($commissionRate / 100);
        echo "  Commission Amount: \${$commission}\n";
    }
    echo "\n";
}

// Example 7: Bulk Processing with Custom Fields
echo "=== Example 7: Bulk Processing with Custom Fields ===\n\n";

$bulkResult = $salesforce->query()->execute(
    "SELECT Id, Name, Industry, CustomerTier__c, LifetimeValue__c, Active__c
     FROM Account
     WHERE Active__c = true
     LIMIT 5"
);

echo "Processing {$bulkResult->totalSize} accounts:\n\n";

$tierStats = [];
foreach ($bulkResult->records as $accountData) {
    $account = Account::fromArray($accountData);

    $tier = $account->getCustomField('CustomerTier__c') ?? 'Unknown';
    $tierStats[$tier] = ($tierStats[$tier] ?? 0) + 1;

    echo "- {$account->name} ({$tier})\n";
}

echo "\nTier Distribution:\n";
foreach ($tierStats as $tier => $count) {
    echo "  {$tier}: {$count}\n";
}

// Example 8: Checking for Custom Field Existence
echo "\n=== Example 8: Safe Custom Field Access ===\n\n";

$testAccount = Account::fromArray([
    'Id' => '001xx000003TEST',
    'Name' => 'Test Account',
    'SomeCustomField__c' => 'exists',
]);

// Safe access with null check
$existingField = $testAccount->getCustomField('SomeCustomField__c');
$missingField = $testAccount->getCustomField('NonExistentField__c');

echo "Existing field value: " . ($existingField ?? 'null') . "\n";
echo "Missing field value: " . ($missingField ?? 'null') . "\n";
echo "\n";

// Check if field exists
$customData = $testAccount->getCustomData();
if (array_key_exists('SomeCustomField__c', $customData)) {
    echo "Field 'SomeCustomField__c' exists\n";
}

if (!array_key_exists('NonExistentField__c', $customData)) {
    echo "Field 'NonExistentField__c' does not exist\n";
}

echo "\n=== Summary ===\n\n";
echo "Benefits of customData approach:\n";
echo "1. No need to extend models for every custom field\n";
echo "2. Works with any Salesforce org's custom fields\n";
echo "3. Dynamic and flexible for varying configurations\n";
echo "4. Still allows typed properties when needed (extend the model)\n";
echo "5. Automatic separation of standard and custom fields\n";
echo "6. Custom fields are automatically included in toArray() for updates\n";
