# Symfony Serializer Integration

This library uses Symfony Serializer for automatic model hydration, providing a powerful and flexible way to map Salesforce API responses to PHP objects.

## Overview

The `BaseModel` class integrates Symfony Serializer to automatically:
- **Denormalize** API response arrays into typed PHP objects
- **Normalize** PHP objects back to arrays
- Handle type conversions (strings, integers, floats, booleans)
- Map field names using annotations
- Skip null values during serialization

## Basic Usage

### Simple Model with Annotations

```php
use SalesforceRestApi\Models\BaseModel;
use Symfony\Component\Serializer\Attribute\SerializedName;

class Contact extends BaseModel
{
    #[SerializedName('Id')]
    public ?string $id = null;

    #[SerializedName('FirstName')]
    public ?string $firstName = null;

    #[SerializedName('LastName')]
    public ?string $lastName = null;

    #[SerializedName('Email')]
    public ?string $email = null;
}

// Automatic hydration
$contact = Contact::fromArray([
    'Id' => '003xx000004TmiXXXX',
    'FirstName' => 'John',
    'LastName' => 'Doe',
    'Email' => 'john@example.com',
]);

echo $contact->firstName; // "John"
```

## Key Features

### 1. Automatic Type Conversion

Symfony Serializer automatically converts types based on property declarations:

```php
class Account extends BaseModel
{
    #[SerializedName('AnnualRevenue')]
    public ?float $annualRevenue = null; // String from API → float

    #[SerializedName('NumberOfEmployees')]
    public ?int $numberOfEmployees = null; // String from API → int

    #[SerializedName('IsActive')]
    public ?bool $isActive = null; // String "true"/"false" → bool
}
```

### 2. Field Name Mapping

Use `#[SerializedName]` to map between Salesforce field names (PascalCase) and PHP properties (camelCase):

```php
class Lead extends BaseModel
{
    #[SerializedName('Id')]
    public ?string $id = null;

    #[SerializedName('FirstName')]
    public ?string $firstName = null; // Maps "FirstName" → $firstName

    #[SerializedName('Company')]
    public ?string $companyName = null; // Maps "Company" → $companyName
}
```

### 3. Custom Hydration Logic

Override `hydrate()` for custom processing after Symfony Serializer completes:

```php
class Account extends BaseModel
{
    #[SerializedName('Id')]
    public ?string $id = null;

    #[SerializedName('Website')]
    public ?string $website = null;

    protected function hydrate(array $data): void
    {
        // Custom logic after Symfony Serializer
        if ($this->website && !str_starts_with($this->website, 'http')) {
            $this->website = 'https://' . $this->website;
        }
    }
}
```

### 4. Serialization Back to Array

Convert models back to arrays with `toArray()`:

```php
$contact = Contact::fromArray(['FirstName' => 'John', 'LastName' => 'Doe']);

$array = $contact->toArray();
// Returns: ['firstName' => 'John', 'lastName' => 'Doe']
// Note: Null values are automatically skipped
```

## Advanced Patterns

### Complex Models with Business Logic

```php
class Opportunity extends BaseModel
{
    #[SerializedName('Id')]
    public ?string $id = null;

    #[SerializedName('Name')]
    public ?string $name = null;

    #[SerializedName('Amount')]
    public ?float $amount = null;

    #[SerializedName('Probability')]
    public ?int $probability = null;

    #[SerializedName('StageName')]
    public ?string $stageName = null;

    #[SerializedName('CloseDate')]
    public ?string $closeDate = null;

    // Computed properties
    public function getExpectedRevenue(): float
    {
        if ($this->amount === null || $this->probability === null) {
            return 0.0;
        }

        return $this->amount * ($this->probability / 100);
    }

    public function isClosingSoon(): bool
    {
        if ($this->closeDate === null) {
            return false;
        }

        $closeDate = new \DateTime($this->closeDate);
        $thirtyDaysFromNow = new \DateTime('+30 days');

        return $closeDate <= $thirtyDaysFromNow;
    }

    public function isWon(): bool
    {
        return $this->stageName === 'Closed Won';
    }
}
```

### Nested Models

For nested objects, handle them in the `hydrate()` method:

```php
class AccountWithContacts extends BaseModel
{
    #[SerializedName('Id')]
    public ?string $id = null;

    #[SerializedName('Name')]
    public ?string $name = null;

    /**
     * @var array<int, Contact>
     */
    public array $contacts = [];

    protected function hydrate(array $data): void
    {
        // Handle nested Contacts relationship
        if (isset($data['Contacts']['records']) && is_array($data['Contacts']['records'])) {
            $this->contacts = array_map(
                fn(array $contactData) => Contact::fromArray($contactData),
                $data['Contacts']['records']
            );
        }
    }
}
```

### Custom Query Results

```php
class OpportunityQueryResult extends BaseModel
{
    public int $totalSize = 0;
    public bool $done = true;

    /**
     * @var array<int, Opportunity>
     */
    public array $records = [];

    protected function hydrate(array $data): void
    {
        if (isset($data['records']) && is_array($data['records'])) {
            $this->records = array_map(
                fn(array $record) => Opportunity::fromArray($record),
                $data['records']
            );
        }
    }

    public function getTotalExpectedRevenue(): float
    {
        return array_reduce(
            $this->records,
            fn(float $sum, Opportunity $opp) => $sum + $opp->getExpectedRevenue(),
            0.0
        );
    }

    public function getOpportunitiesClosingSoon(): array
    {
        return array_filter(
            $this->records,
            fn(Opportunity $opp) => $opp->isClosingSoon()
        );
    }
}
```

## Serializer Configuration

### Custom Serializer Instance

You can provide a custom configured serializer:

```php
use Symfony\Component\Serializer\Encoder\JsonEncoder;
use Symfony\Component\Serializer\Normalizer\DateTimeNormalizer;
use Symfony\Component\Serializer\Normalizer\ObjectNormalizer;
use Symfony\Component\Serializer\Serializer;

$normalizers = [
    new DateTimeNormalizer(),
    new ObjectNormalizer(),
];
$encoders = [new JsonEncoder()];

$customSerializer = new Serializer($normalizers, $encoders);

// Set globally for all models
BaseModel::setSerializer($customSerializer);
```

### Per-Model Serializer

```php
class CustomModel extends BaseModel
{
    protected static function getSerializer(): SerializerInterface
    {
        // Return a custom serializer just for this model
        $normalizers = [
            new DateTimeNormalizer(['datetime_format' => 'Y-m-d']),
            new ObjectNormalizer(),
        ];

        return new Serializer($normalizers, [new JsonEncoder()]);
    }
}
```

## Benefits

### 1. Less Boilerplate Code

**Before (Manual Hydration):**
```php
protected function hydrate(array $data): void
{
    $this->id = $data['Id'] ?? null;
    $this->firstName = $data['FirstName'] ?? null;
    $this->lastName = $data['LastName'] ?? null;
    $this->email = $data['Email'] ?? null;
    $this->phone = $data['Phone'] ?? null;
    $this->title = $data['Title'] ?? null;
    $this->department = $data['Department'] ?? null;
    // ... 20 more fields
}
```

**After (Symfony Serializer):**
```php
// Just declare properties with attributes - that's it!
#[SerializedName('FirstName')]
public ?string $firstName = null;

#[SerializedName('LastName')]
public ?string $lastName = null;

// Optional: Only for custom logic
protected function hydrate(array $data): void
{
    // Custom processing only
}
```

### 2. Automatic Type Safety

Symfony Serializer handles type conversions automatically, reducing errors:

```php
// API returns: "100000.50" (string)
#[SerializedName('AnnualRevenue')]
public ?float $annualRevenue = null; // Automatically converted to float

// API returns: "500" (string)
#[SerializedName('NumberOfEmployees')]
public ?int $numberOfEmployees = null; // Automatically converted to int
```

### 3. Bidirectional Serialization

Easily convert back to arrays for updates:

```php
$account = $salesforce->sobject('Account')->get($id, Account::class);
$account->phone = '555-1234';
$account->website = 'https://example.com';

// Convert to array for API update
$updateData = $account->toArray();
$salesforce->sobject('Account')->update($id, $updateData);
```

## Common Patterns

### Pattern 1: Value Objects

```php
class Money extends BaseModel
{
    public ?float $amount = null;
    public ?string $currency = null;

    public function format(): string
    {
        return sprintf('%s %.2f', $this->currency, $this->amount);
    }
}

class Opportunity extends BaseModel
{
    public ?Money $amount = null;

    protected function hydrate(array $data): void
    {
        if (isset($data['Amount'], $data['CurrencyIsoCode'])) {
            $this->amount = Money::fromArray([
                'amount' => $data['Amount'],
                'currency' => $data['CurrencyIsoCode'],
            ]);
        }
    }
}
```

### Pattern 2: Date Handling

```php
class Task extends BaseModel
{
    #[SerializedName('Id')]
    public ?string $id = null;

    #[SerializedName('Subject')]
    public ?string $subject = null;

    #[SerializedName('ActivityDate')]
    public ?string $activityDate = null;

    private ?\DateTime $activityDateTime = null;

    protected function hydrate(array $data): void
    {
        if ($this->activityDate) {
            $this->activityDateTime = new \DateTime($this->activityDate);
        }
    }

    public function isOverdue(): bool
    {
        if ($this->activityDateTime === null) {
            return false;
        }

        return $this->activityDateTime < new \DateTime();
    }
}
```

### Pattern 3: Validation

```php
class Lead extends BaseModel
{
    #[SerializedName('Email')]
    public ?string $email = null;

    #[SerializedName('Phone')]
    public ?string $phone = null;

    #[SerializedName('Company')]
    public ?string $company = null;

    protected function hydrate(array $data): void
    {
        // Validate and normalize data
        if ($this->email) {
            $this->email = strtolower(trim($this->email));
        }

        if ($this->phone) {
            // Remove non-numeric characters
            $this->phone = preg_replace('/[^0-9]/', '', $this->phone);
        }
    }

    public function isValid(): bool
    {
        return $this->email !== null
            && filter_var($this->email, FILTER_VALIDATE_EMAIL) !== false
            && $this->company !== null;
    }
}
```

## Best Practices

1. **Use Annotations for Field Mapping**: Always use `#[SerializedName]` for clarity
2. **Reserve hydrate() for Custom Logic**: Let Symfony Serializer handle basic mapping
3. **Type Your Properties**: Use proper type hints for automatic conversion
4. **Document Complex Types**: Use PHPDoc for arrays and custom types
5. **Keep Models Focused**: One model per SObject type
6. **Add Business Logic**: Use models for computed properties and validation

## Migration Guide

If you have existing models without Symfony Serializer:

```php
// Before
class Account extends BaseModel
{
    public ?string $name = null;

    protected function hydrate(array $data): void
    {
        $this->name = $data['Name'] ?? null;
    }
}

// After
class Account extends BaseModel
{
    #[SerializedName('Name')]
    public ?string $name = null;

    // hydrate() can be removed if no custom logic needed
}
```

No breaking changes - existing models continue to work!
