# Salesforce Standard Object Models

This library includes pre-built models for common Salesforce standard objects. All models use Symfony Serializer for automatic hydration and include computed properties for common operations.

## Available Models

### Sales Objects

#### Account
`SalesforceRestApi\Models\Standard\Account`

Represents an organization or person involved with your business.

**Key Fields:**
- Name, Type, Industry
- AnnualRevenue, NumberOfEmployees
- Phone, Website, Fax
- BillingAddress (Street, City, State, PostalCode, Country)
- ShippingAddress (Street, City, State, PostalCode, Country)
- OwnerId, ParentId

**Computed Properties:**
```php
$account->isEnterprise(): bool  // >= 1000 employees
$account->getRevenuePerEmployee(): ?float
```

**Example:**
```php
use SalesforceRestApi\Models\Standard\Account;

$account = $salesforce->sobject('Account')->get($id, Account::class);
echo "Revenue per employee: $" . number_format($account->getRevenuePerEmployee());
```

#### Contact
`SalesforceRestApi\Models\Standard\Contact`

Represents an individual associated with an account.

**Key Fields:**
- FirstName, LastName, Salutation, Suffix, Title
- Email, Phone, MobilePhone, HomePhone
- AccountId, ReportsToId
- MailingAddress, OtherAddress
- Department, Birthdate
- HasOptedOutOfEmail, DoNotCall

**Computed Properties:**
```php
$contact->getFullName(): string
$contact->getMailingAddress(): string
$contact->isEmailValid(): bool
$contact->canContact(): bool
```

**Example:**
```php
use SalesforceRestApi\Models\Standard\Contact;

$contact = $salesforce->sobject('Contact')->get($id, Contact::class);
echo $contact->getFullName();

if ($contact->canContact() && $contact->isEmailValid()) {
    // Safe to send email
}
```

#### Lead
`SalesforceRestApi\Models\Standard\Lead`

Represents a prospect or potential sales opportunity.

**Key Fields:**
- FirstName, LastName, Company, Title
- Email, Phone, MobilePhone, Website
- Status, Rating, LeadSource
- Industry, AnnualRevenue, NumberOfEmployees
- IsConverted, ConvertedAccountId, ConvertedContactId
- Street, City, State, PostalCode, Country

**Computed Properties:**
```php
$lead->getFullName(): string
$lead->getAddress(): string
$lead->isQualified(): bool
$lead->canContact(): bool
$lead->isEmailValid(): bool
```

**Example:**
```php
use SalesforceRestApi\Models\Standard\Lead;

$lead = $salesforce->sobject('Lead')->get($id, Lead::class);

if ($lead->isQualified() && !$lead->isConverted) {
    // Ready for conversion
}
```

#### Opportunity
`SalesforceRestApi\Models\Standard\Opportunity`

Represents a sale or pending deal.

**Key Fields:**
- Name, StageName, Amount, Probability
- CloseDate, ExpectedRevenue
- AccountId, ContactId, OwnerId
- Type, LeadSource, NextStep
- IsClosed, IsWon, ForecastCategory

**Computed Properties:**
```php
$opportunity->getExpectedRevenue(): float
$opportunity->isClosingSoon(int $daysThreshold = 30): bool
$opportunity->isOverdue(): bool
$opportunity->getDaysUntilClose(): ?int
$opportunity->isHighValue(float $threshold = 100000): bool
$opportunity->getStageProgress(): ?int
```

**Example:**
```php
use SalesforceRestApi\Models\Standard\Opportunity;

$opp = $salesforce->sobject('Opportunity')->get($id, Opportunity::class);

if ($opp->isClosingSoon(7)) {
    echo "Closes in " . $opp->getDaysUntilClose() . " days";
    echo "Expected revenue: $" . number_format($opp->getExpectedRevenue());
}
```

#### OpportunityLineItem
`SalesforceRestApi\Models\Standard\OpportunityLineItem`

Represents a product on an opportunity.

**Key Fields:**
- OpportunityId, Product2Id, PricebookEntryId
- Quantity, UnitPrice, TotalPrice
- ListPrice, Discount
- ProductCode, Name, Description

**Computed Properties:**
```php
$lineItem->getDiscountPercentage(): ?float
$lineItem->getSubtotal(): ?float
```

#### OpportunityContactRole
`SalesforceRestApi\Models\Standard\OpportunityContactRole`

Represents the role of a contact on an opportunity.

**Key Fields:**
- OpportunityId, ContactId
- Role, IsPrimary

**Computed Properties:**
```php
$role->isDecisionMaker(): bool
$role->isInfluencer(): bool
```

**Example:**
```php
use SalesforceRestApi\Models\Standard\OpportunityContactRole;

$role = $salesforce->sobject('OpportunityContactRole')->get($id, OpportunityContactRole::class);

if ($role->isPrimary && $role->isDecisionMaker()) {
    echo "Primary decision maker on this opportunity";
}
```

### Purchase Objects

#### Purchase
`SalesforceRestApi\Models\Standard\Purchase`

Represents a completed transaction.

**Key Fields:**
- Name, PurchaseNumber, Status
- TotalAmount, TaxAmount, GrandTotalAmount
- PurchaseDate, ExpectedDeliveryDate, ActualDeliveryDate
- AccountId, BillingAddress, ShippingAddress
- PaymentMethod, PaymentStatus, PaymentTerms
- TrackingNumber, Carrier

**Computed Properties:**
```php
$purchase->getTaxPercentage(): ?float
$purchase->isDelivered(): bool
$purchase->isOverdueDelivery(): bool
$purchase->getDaysUntilDelivery(): ?int
$purchase->isPaid(): bool
$purchase->isPending(): bool
$purchase->isCompleted(): bool
```

**Example:**
```php
use SalesforceRestApi\Models\Standard\Purchase;

$purchase = $salesforce->sobject('Purchase')->get($id, Purchase::class);

if (!$purchase->isDelivered() && $purchase->isOverdueDelivery()) {
    echo "Delivery is overdue by " . abs($purchase->getDaysUntilDelivery()) . " days";
}

echo "Tax Rate: " . number_format($purchase->getTaxPercentage(), 2) . "%";
echo "Payment Status: " . ($purchase->isPaid() ? 'Paid' : 'Pending');
```

#### PurchaseLineItem
`SalesforceRestApi\Models\Standard\PurchaseLineItem`

Represents an individual product or service on a purchase.

**Key Fields:**
- PurchaseId, Product2Id, ProductCode
- Quantity, UnitPrice, ListPrice
- TotalLineAmount, Discount, DiscountPercentage
- TaxAmount, TaxPercentage, GrandTotalAmount
- LineNumber, Status, DeliveryDate

**Computed Properties:**
```php
$item->getSubtotal(): ?float
$item->getDiscountAmount(): ?float
$item->getEffectiveDiscountPercentage(): ?float
$item->getTotalWithTax(): ?float
$item->getEffectiveTaxPercentage(): ?float
$item->getSavings(): ?float
$item->getSavingsPercentage(): ?float
```

**Example:**
```php
use SalesforceRestApi\Models\Standard\PurchaseLineItem;

$item = $salesforce->sobject('PurchaseLineItem')->get($id, PurchaseLineItem::class);

echo "Subtotal: $" . number_format($item->getSubtotal());
echo "Discount: " . number_format($item->getEffectiveDiscountPercentage(), 2) . "%";
echo "Savings: $" . number_format($item->getSavings());
echo "Total with Tax: $" . number_format($item->getTotalWithTax());
```

### Service Objects

#### Case
`SalesforceRestApi\Models\Standard\CaseModel`

Represents a customer issue or problem.

**Key Fields:**
- CaseNumber, Subject, Description, Status
- Priority, Reason, Origin, Type
- AccountId, ContactId
- IsClosed, ClosedDate, IsEscalated
- ContactEmail, ContactPhone

**Computed Properties:**
```php
$case->getAge(): ?int
$case->getResolutionTime(): ?int
$case->isOpen(): bool
$case->isHighPriority(): bool
```

**Example:**
```php
use SalesforceRestApi\Models\Standard\CaseModel;

$case = $salesforce->sobject('Case')->get($id, CaseModel::class);

if ($case->isOpen() && $case->getAge() > 7) {
    echo "Case has been open for " . $case->getAge() . " days";
}
```

### Activity Objects

#### Task
`SalesforceRestApi\Models\Standard\Task`

Represents a to-do item or activity.

**Key Fields:**
- Subject, Description, Status, Priority
- ActivityDate, WhoId, WhatId
- OwnerId, IsReminderSet
- CallDurationInSeconds, CallType

**Computed Properties:**
```php
$task->isOverdue(): bool
$task->isClosed(): bool
$task->getDaysUntilDue(): ?int
$task->getCallDurationMinutes(): ?float
```

**Example:**
```php
use SalesforceRestApi\Models\Standard\Task;

$task = $salesforce->sobject('Task')->get($id, Task::class);

if ($task->isOverdue() && !$task->isClosed()) {
    echo "Overdue by " . abs($task->getDaysUntilDue()) . " days";
}
```

#### Event
`SalesforceRestApi\Models\Standard\Event`

Represents a calendar appointment.

**Key Fields:**
- Subject, Description, Location, Type
- StartDateTime, EndDateTime, DurationInMinutes
- WhoId, WhatId, OwnerId
- IsAllDayEvent, IsPrivate, ShowAs

**Computed Properties:**
```php
$event->isUpcoming(): bool
$event->isPast(): bool
$event->isOngoing(): bool
$event->getDurationHours(): ?float
$event->getTimeUntilStart(): ?int
```

### System Objects

#### User
`SalesforceRestApi\Models\Standard\User`

Represents a Salesforce user.

**Key Fields:**
- Username, FirstName, LastName, Email
- Title, Department, Division
- ProfileId, UserRoleId, ManagerId
- IsActive, LastLoginDate
- TimeZoneSidKey, LanguageLocaleKey

**Computed Properties:**
```php
$user->getFullName(): string
$user->getAddress(): string
$user->getDaysSinceLastLogin(): ?int
```

### Marketing Objects

#### Campaign
`SalesforceRestApi\Models\Standard\Campaign`

Represents a marketing project.

**Key Fields:**
- Name, Type, Status
- StartDate, EndDate
- BudgetedCost, ActualCost, ExpectedRevenue
- NumberOfLeads, NumberOfConvertedLeads
- NumberOfOpportunities, NumberOfWonOpportunities
- IsActive

**Computed Properties:**
```php
$campaign->getROI(): ?float
$campaign->getConversionRate(): ?float
$campaign->getResponseRate(): ?float
$campaign->isOngoing(): bool
$campaign->isPast(): bool
```

**Example:**
```php
use SalesforceRestApi\Models\Standard\Campaign;

$campaign = $salesforce->sobject('Campaign')->get($id, Campaign::class);

echo "ROI: " . number_format($campaign->getROI(), 2) . "%";
echo "Conversion Rate: " . number_format($campaign->getConversionRate(), 2) . "%";
```

### Product Objects

#### Product2
`SalesforceRestApi\Models\Standard\Product2`

Represents a product.

**Key Fields:**
- Name, ProductCode, Description
- Family, IsActive
- QuantityUnitOfMeasure

#### Pricebook2
`SalesforceRestApi\Models\Standard\Pricebook2`

Represents a price book.

**Key Fields:**
- Name, Description
- IsActive, IsStandard

### Content Objects

#### Note
`SalesforceRestApi\Models\Standard\Note`

Represents a note attached to a record.

**Key Fields:**
- ParentId, Title, Body
- IsPrivate, OwnerId

#### Attachment
`SalesforceRestApi\Models\Standard\Attachment`

Represents a file attachment.

**Key Fields:**
- ParentId, Name, ContentType
- BodyLength, Body, Description
- IsPrivate

**Computed Properties:**
```php
$attachment->getFileSizeKB(): ?float
$attachment->getFileSizeMB(): ?float
```

## Usage Examples

### Basic CRUD with Standard Models

```php
use SalesforceRestApi\Models\Standard\Account;
use SalesforceRestApi\Models\Standard\Contact;
use SalesforceRestApi\Salesforce;

$sf = Salesforce::withPassword(...);

// Create Account
$accountData = [
    'Name' => 'Acme Corp',
    'Industry' => 'Technology',
    'AnnualRevenue' => 5000000,
    'NumberOfEmployees' => 150,
];

$result = $sf->sobject('Account')->create($accountData);
$accountId = $result['id'];

// Fetch with model
$account = $sf->sobject('Account')->get($accountId, Account::class);

echo "Company: " . $account->name;
echo "Enterprise: " . ($account->isEnterprise() ? 'Yes' : 'No');
echo "Revenue/Employee: $" . number_format($account->getRevenuePerEmployee());
```

### Querying with Standard Models

```php
use SalesforceRestApi\Models\Standard\Opportunity;
use SalesforceRestApi\Models\QueryResult;

// Query opportunities closing soon
$result = $sf->query()->execute(
    "SELECT Id, Name, Amount, CloseDate, StageName, Probability
     FROM Opportunity
     WHERE IsClosed = false
     AND CloseDate <= NEXT_N_DAYS:30"
);

foreach ($result->getRecords() as $record) {
    // Convert to strongly-typed model
    $opp = Opportunity::fromArray($record->getRawData());

    if ($opp->isClosingSoon(7)) {
        echo "{$opp->name}: Closes in {$opp->getDaysUntilClose()} days\n";
        echo "Expected Revenue: $" . number_format($opp->getExpectedRevenue()) . "\n";
    }
}
```

### Custom Query Results with Standard Models

```php
use SalesforceRestApi\Models\BaseModel;
use SalesforceRestApi\Models\Standard\Contact;

class ContactQueryResult extends BaseModel
{
    public int $totalSize = 0;
    public bool $done = true;
    public array $records = [];

    protected function hydrate(array $data): void
    {
        if (isset($data['records'])) {
            $this->records = array_map(
                fn(array $r) => Contact::fromArray($r),
                $data['records']
            );
        }
    }

    /**
     * @return Contact[]
     */
    public function getContacts(): array
    {
        return $this->records;
    }

    public function getContactableContacts(): array
    {
        return array_filter(
            $this->records,
            fn(Contact $c) => $c->canContact() && $c->isEmailValid()
        );
    }
}

$result = $sf->query()->execute(
    "SELECT Id, FirstName, LastName, Email, HasOptedOutOfEmail, DoNotCall
     FROM Contact
     LIMIT 100",
    ContactQueryResult::class
);

$contactable = $result->getContactableContacts();
echo "Contactable: " . count($contactable) . " of " . $result->totalSize;
```

### Batch Operations

```php
use SalesforceRestApi\Models\Standard\Lead;

// Get all qualified leads
$result = $sf->query()->execute(
    "SELECT Id, FirstName, LastName, Company, Email, Status
     FROM Lead
     WHERE Status = 'Qualified' AND IsConverted = false"
);

foreach ($result->getRecords() as $record) {
    $lead = Lead::fromArray($record->getRawData());

    if ($lead->isQualified() && !$lead->isConverted) {
        echo "Ready to convert: " . $lead->getFullName() . " at " . $lead->company . "\n";
    }
}
```

## Benefits

1. **Type Safety**: All fields are properly typed with PHP 8.4 type hints
2. **Intellisense**: Full IDE autocomplete support
3. **Business Logic**: Built-in computed properties for common operations
4. **Validation**: Methods like `isEmailValid()`, `canContact()`
5. **Consistency**: Standardized field names and access patterns
6. **Extensibility**: Extend any model with custom properties and methods

## Extending Standard Models

You can extend any standard model with custom fields:

```php
use SalesforceRestApi\Models\Standard\Account;
use Symfony\Component\Serializer\Attribute\SerializedName;

class MyAccount extends Account
{
    #[SerializedName('CustomField__c')]
    public ?string $customField = null;

    #[SerializedName('AnotherCustomField__c')]
    public ?int $anotherCustomField = null;

    public function getCustomLogic(): string
    {
        return "Custom: " . $this->customField;
    }
}

$account = $sf->sobject('Account')->get($id, MyAccount::class);
echo $account->getCustomLogic();
```

## Complete Model List

| Model | Object | Namespace |
|-------|--------|-----------|
| Account | Account | SalesforceRestApi\Models\Standard\Account |
| Contact | Contact | SalesforceRestApi\Models\Standard\Contact |
| Lead | Lead | SalesforceRestApi\Models\Standard\Lead |
| Opportunity | Opportunity | SalesforceRestApi\Models\Standard\Opportunity |
| OpportunityLineItem | OpportunityLineItem | SalesforceRestApi\Models\Standard\OpportunityLineItem |
| OpportunityContactRole | OpportunityContactRole | SalesforceRestApi\Models\Standard\OpportunityContactRole |
| Purchase | Purchase | SalesforceRestApi\Models\Standard\Purchase |
| PurchaseLineItem | PurchaseLineItem | SalesforceRestApi\Models\Standard\PurchaseLineItem |
| CaseModel | Case | SalesforceRestApi\Models\Standard\CaseModel |
| Task | Task | SalesforceRestApi\Models\Standard\Task |
| Event | Event | SalesforceRestApi\Models\Standard\Event |
| User | User | SalesforceRestApi\Models\Standard\User |
| Campaign | Campaign | SalesforceRestApi\Models\Standard\Campaign |
| Product2 | Product2 | SalesforceRestApi\Models\Standard\Product2 |
| Pricebook2 | Pricebook2 | SalesforceRestApi\Models\Standard\Pricebook2 |
| Note | Note | SalesforceRestApi\Models\Standard\Note |
| Attachment | Attachment | SalesforceRestApi\Models\Standard\Attachment |
