# Salesforce REST API PHP Library

A modern PHP 8.4 library for interacting with the Salesforce REST API, built with Guzzle 7 and featuring extensible model support.

## Features

- **PHP 8.4 Compatible**: Leverages modern PHP features including typed properties, named arguments, and readonly properties
- **Guzzle 7 HTTP Client**: Reliable and well-tested HTTP client
- **Symfony Serializer Integration**: Automatic model hydration using Symfony Serializer with annotation support
- **Pre-built Standard Models**: 17 ready-to-use models for common Salesforce objects (Account, Contact, Lead, Opportunity, Purchase, etc.)
- **Extensible Models**: Pass custom model classes to receive and extend API responses
- **Full CRUD Operations**: Complete support for SObject create, read, update, and delete operations
- **Query & Search**: Execute SOQL queries and SOSL searches with automatic pagination
- **Multiple Auth Methods**: Support for OAuth 2.0 password flow, client credentials, refresh tokens, and authorization codes
- **Type Safety**: Comprehensive type hints and PHPDoc annotations for better IDE support
- **Exception Handling**: Dedicated exception classes for better error handling

## Installation

```bash
composer require your-vendor/salesforce-rest-api
```

## Requirements

- PHP 8.4 or higher
- ext-json
- guzzlehttp/guzzle ^7.0
- symfony/serializer ^7.0
- symfony/property-access ^7.0

## Quick Start

```php
use SalesforceRestApi\Salesforce;

// Authenticate
$salesforce = Salesforce::withPassword(
    clientId: 'your_client_id',
    clientSecret: 'your_client_secret',
    username: 'your_username@example.com',
    password: 'your_password',
    securityToken: 'your_security_token' // Optional if IP is whitelisted
);

// Create a record
$account = $salesforce->sobject('Account');
$result = $account->createFromArray([
    'Name' => 'Acme Corporation',
    'Industry' => 'Technology',
]);

// Read a record
$accountRecord = $account->get($result['id']);
echo $accountRecord->getField('Name');

// Update a record
$account->update($result['id'], [
    'Phone' => '555-1234',
]);

// Delete a record
$account->delete($result['id']);
```

## Standard Models

The library includes pre-built models for 17 common Salesforce objects with full type safety and computed properties.

### Using Standard Models

```php
use SalesforceRestApi\Models\Standard\Account;
use SalesforceRestApi\Models\Standard\Contact;
use SalesforceRestApi\Models\Standard\Opportunity;

// Fetch with automatic hydration
$account = $salesforce->sobject('Account')->get($id, Account::class);

echo "Company: {$account->name}\n";
echo "Revenue/Employee: $" . number_format($account->getRevenuePerEmployee());
echo "Is Enterprise: " . ($account->isEnterprise() ? 'Yes' : 'No');
```

### Available Models

- **Sales**: Account, Contact, Lead, Opportunity, OpportunityLineItem, OpportunityContactRole
- **Purchase**: Purchase, PurchaseLineItem
- **Service**: CaseModel (Case)
- **Activities**: Task, Event
- **Marketing**: Campaign
- **Products**: Product2, Pricebook2
- **System**: User
- **Content**: Note, Attachment

Each model includes:
- All standard Salesforce fields with proper types
- Computed properties for common operations
- Helper methods for business logic

See [STANDARD_MODELS.md](STANDARD_MODELS.md) for complete documentation.

## Authentication

### Username/Password Flow

```php
$salesforce = Salesforce::withPassword(
    clientId: 'your_client_id',
    clientSecret: 'your_client_secret',
    username: 'your_username@example.com',
    password: 'your_password',
    securityToken: 'your_security_token'
);
```

### Client Credentials Flow

```php
$salesforce = Salesforce::withClientCredentials(
    clientId: 'your_client_id',
    clientSecret: 'your_client_secret'
);
```

### Existing Access Token

```php
$salesforce = Salesforce::withAccessToken([
    'access_token' => 'your_access_token',
    'instance_url' => 'https://yourinstance.salesforce.com',
    'token_type' => 'Bearer',
]);
```

### Sandbox Environment

```php
$salesforce = Salesforce::withPassword(
    clientId: 'your_client_id',
    clientSecret: 'your_client_secret',
    username: 'your_username@example.com.sandbox',
    password: 'your_password',
    loginUrl: 'https://test.salesforce.com'
);
```

## SObject Operations

### Create

```php
$contact = $salesforce->sobject('Contact');
$result = $contact->createFromArray([
    'FirstName' => 'John',
    'LastName' => 'Doe',
    'Email' => 'john.doe@example.com',
]);

echo "Created Contact ID: " . $result['id'];
```

### Read

```php
// Get all fields
$contact = $salesforce->sobject('Contact')->get('003xx000004TmiXXXX');

// Get specific fields
$contact = $salesforce->sobject('Contact')->get(
    '003xx000004TmiXXXX',
    fields: ['FirstName', 'LastName', 'Email']
);

echo $contact->getField('Email');
```

### Update

```php
$salesforce->sobject('Contact')->update('003xx000004TmiXXXX', [
    'Email' => 'new.email@example.com',
]);
```

### Delete

```php
$salesforce->sobject('Contact')->delete('003xx000004TmiXXXX');
```

### Upsert

```php
$result = $salesforce->sobject('Contact')->upsert(
    externalIdField: 'External_ID__c',
    externalId: 'EXT-12345',
    data: [
        'FirstName' => 'Jane',
        'LastName' => 'Smith',
    ]
);
```

### Describe

```php
// Get full object metadata
$metadata = $salesforce->sobject('Account')->describe();
echo "Fields: " . count($metadata['fields']);

// Get basic object info
$info = $salesforce->sobject('Account')->metadata();
```

## Queries

### Basic Query

```php
$result = $salesforce->query()->execute(
    "SELECT Id, Name, Email FROM Contact WHERE LastName = 'Smith'"
);

echo "Total: " . $result->getTotalSize() . "\n";

foreach ($result->getRecords() as $record) {
    echo $record->getField('Name') . " - " . $record->getField('Email') . "\n";
}
```

### Query All (Including Deleted)

```php
$result = $salesforce->query()->queryAll(
    "SELECT Id, Name, IsDeleted FROM Account"
);
```

### Automatic Pagination

```php
// Automatically fetches all pages
$allResults = $salesforce->query()->all(
    "SELECT Id, Name FROM Account"
);

foreach ($allResults as $batch) {
    foreach ($batch->getRecords() as $record) {
        // Process each record
    }
}
```

### Manual Pagination

```php
$result = $salesforce->query()->execute("SELECT Id, Name FROM Account");

while (!$result->isDone()) {
    // Process current batch
    foreach ($result->getRecords() as $record) {
        // ...
    }

    // Get next batch
    $result = $salesforce->query()->next($result->getNextRecordsUrl());
}
```

## Search

### SOSL Search

```php
$results = $salesforce->search()->execute(
    "FIND {John Smith} IN NAME FIELDS RETURNING Contact(Id, Name), Lead(Id, Name)"
);
```

### Parameterized Search

```php
$results = $salesforce->search()->parameterized([
    'q' => 'John',
    'sobjects' => [
        ['name' => 'Contact'],
        ['name' => 'Lead'],
    ],
]);
```

## Custom Models

### Using Symfony Serializer (Recommended)

The library uses Symfony Serializer for automatic model hydration. Simply define properties with `SerializedName` attributes:

```php
use SalesforceRestApi\Models\BaseModel;
use Symfony\Component\Serializer\Attribute\SerializedName;

class Account extends BaseModel
{
    #[SerializedName('Id')]
    public ?string $id = null;

    #[SerializedName('Name')]
    public ?string $name = null;

    #[SerializedName('Industry')]
    public ?string $industry = null;

    #[SerializedName('AnnualRevenue')]
    public ?float $annualRevenue = null;

    // Optional: Add custom logic after serialization
    protected function hydrate(array $data): void
    {
        // Example: Normalize website URL
        if ($this->website && !str_starts_with($this->website, 'http')) {
            $this->website = 'https://' . $this->website;
        }
    }

    // Add custom methods
    public function getName(): ?string
    {
        return $this->name;
    }

    // Add custom methods
    public function isTechnologyCompany(): bool
    {
        return $this->industry === 'Technology';
    }
}

// Use custom model
$account = $salesforce->sobject('Account')->get('001xx000003DGbXXXX', Account::class);
echo $account->getName();

if ($account->isTechnologyCompany()) {
    echo "This is a tech company!";
}
```

### Custom Query Result Models

```php
class AccountQueryResult extends BaseModel
{
    public int $totalSize = 0;
    public bool $done = true;

    /**
     * @var array<int, Account>
     */
    public array $records = [];

    // Symfony Serializer handles primitive types automatically
    // Only use hydrate() to convert nested arrays to model instances
    protected function hydrate(array $data): void
    {
        if (isset($data['records']) && is_array($data['records'])) {
            $this->records = array_map(
                fn(array $r) => Account::fromArray($r),
                $data['records']
            );
        }
    }

    public function getAccounts(): array
    {
        return $this->records;
    }
}

$result = $salesforce->query()->execute(
    "SELECT Id, Name, Industry FROM Account",
    AccountQueryResult::class
);

foreach ($result->getAccounts() as $account) {
    echo $account->getName();
}
```

## Additional Operations

### List API Versions

```php
$versions = $salesforce->versions();
foreach ($versions as $version) {
    echo $version['label'] . " - " . $version['version'] . "\n";
}
```

### List Resources

```php
$resources = $salesforce->resources();
print_r($resources);
```

### Describe Global

```php
$global = $salesforce->describeGlobal();
echo "Available SObjects: " . count($global['sobjects']) . "\n";
```

### Get Updated Records

```php
$updated = $salesforce->sobject('Account')->updated(
    startDate: '2025-01-01T00:00:00Z',
    endDate: '2025-01-31T23:59:59Z'
);
```

### Get Deleted Records

```php
$deleted = $salesforce->sobject('Account')->deleted(
    startDate: '2025-01-01T00:00:00Z',
    endDate: '2025-01-31T23:59:59Z'
);
```

## Exception Handling

```php
use SalesforceRestApi\Exceptions\ApiException;
use SalesforceRestApi\Exceptions\AuthenticationException;

try {
    $result = $salesforce->sobject('Account')->createFromArray([
        'Name' => 'Test Account',
    ]);
} catch (AuthenticationException $e) {
    echo "Authentication failed: " . $e->getMessage();
} catch (ApiException $e) {
    echo "API error: " . $e->getMessage();

    // Get error details
    $errorData = $e->getErrorData();
    if ($errorData) {
        print_r($errorData);
    }
}
```

## Advanced Configuration

### Custom API Version

```php
$salesforce = Salesforce::withPassword(
    clientId: 'your_client_id',
    clientSecret: 'your_client_secret',
    username: 'your_username@example.com',
    password: 'your_password',
    apiVersion: 'v60.0'
);
```

### Custom Guzzle Client

```php
use GuzzleHttp\Client;

$httpClient = new Client([
    'timeout' => 30,
    'verify' => true,
]);

$salesforce = Salesforce::withPassword(
    clientId: 'your_client_id',
    clientSecret: 'your_client_secret',
    username: 'your_username@example.com',
    password: 'your_password',
    httpClient: $httpClient
);
```

## License

MIT

## Contributing

Contributions are welcome! Please feel free to submit a Pull Request.

## Support

For issues and questions, please use the GitHub issue tracker.
