<?php

/*
 * Copyright Talisman Innovations Ltd. (2017). All rights reserved.
 */

namespace Revel;

use Psr\Log\LoggerInterface;
use GuzzleHttp\Client;
use Exception;

ini_set('memory_limit', '-1');

class Call {

    protected $client;
    protected $key;
    protected $secret;
    protected $format;
    protected $establishment;
    protected $user;
    protected $limit = 100;

    /**
     * SetUp the Client
     * @param LoggerInterface $logger
     * @param string $shopName
     */
    public function __construct(LoggerInterface $logger, $shopName, $key, $secret, $establishment = null, $user = null, $format = 'json') {
        $this->client = new Client(['base_uri' => 'https://' . $shopName . '.revelup.com/']);
        $this->key = $key;
        $this->secret = $secret;
        $this->format = $format;
        $this->establishment = $establishment;
        $this->user = $user;
        $this->logger = $logger;
    }

    /**
     * Post data
     * @param type $endpoint
     * @param array $data
     * @return type
     * @throws Exception
     */
    public function post($endpoint, array $data) {
        
        $urlParts = '?';
        if ($this->format) {
            $urlParts .= 'format=' . $this->format;
        }
        if ($this->establishment) {
            $urlParts .= '&establishment=' . $this->establishment;
        }
        if ($this->user) {
            $urlParts .= '&user=' . $this->user;
        }
        
        $response = $this->client->request('POST', $endpoint . $urlParts, ['json'=>$data, 'headers' => ['API-AUTHENTICATION' => $this->key . ':' . $this->secret]]);
        $code = $response->getStatusCode(); // 200
        $reason = $response->getReasonPhrase(); // OK

        if ($this->format == 'json') {
            $return = json_decode($response->getBody());
        } else {
            $return = $response->getBody();
        }
        
        $ok = ($code == 200 || $code == 201)?1:0;

        if (!$ok) {
           $this->logger->critical('Error '.$return->error,(array)$response->getBody());
           throw new Exception('ERROR status code ' . $code . ' Reason: '.$reason);
        }
        if (isset($return->status ) && $return->status == 'ERROR') {
           $this->logger->critical('Error ',(array)$response->getBody());
           throw new Exception('ERROR ' . $return->error->message);
        }
        
        
        return $return;
    }
    
    
    /**
     * Create and send the standard request default limit 20
     * use for single result calls
     * @param string $endpoint
     * @return object
     */
    public function request($endpoint, $noEstablishment = null, array $singleOpt = null) {
        
        $this->logger->debug('request');
        
        $urlParts = '?';
        if ($this->format) {
            $urlParts .= 'format=' . $this->format;
        }
        if ($this->establishment && !$noEstablishment) {
            $urlParts .= '&establishment=' . $this->establishment;
        }
        if ($this->user) {
            $urlParts .= '&user=' . $this->user;
        }
        if($singleOpt) {
            foreach($singleOpt as $k=>$v) {
                $urlParts .= '&'.$k.'='.$v;
            }
        }
        
        $response = $this->client->request('GET', $endpoint . $urlParts, ['headers' => ['API-AUTHENTICATION' => $this->key . ':' . $this->secret]]);
        $code = $response->getStatusCode(); // 200
        $reason = $response->getReasonPhrase(); // OK
        
        if ($code != 200) {
           throw new Exception('ERROR status code ' . $code . ' Reason: '.$reason);
        }

        $this->logger->debug('response',(array)json_decode($response->getBody()));
        
        if ($this->format == 'json') {
            $return = json_decode($response->getBody());
        } else {
            $return = $response->getBody();
        }

        return $return;
    }
    
    /**
     * Get all objects paginates through up to $limit at a time.
     * @param string $endpoint
     * @param array $search
     * @return array of objects
     */
    public function requestAll($endpoint, array $search = [], $noEstablishment = null) {

        $this->logger->debug('allit');
        
        $urlParts = '?';
        # Add the data format
        if ($this->format) {
            $urlParts .= 'format=' . $this->format;
        }
        # Add the establishment ID
        if ($this->establishment && !$noEstablishment) {
            $urlParts .= '&establishment=' . $this->establishment;
        }
        # Add User ID
        if ($this->user) {
            $urlParts .= '&user=' . $this->user;
        }
        
        # Setup the search query string
        if (!empty($search)) {
            foreach ($search as $key => $parts) {
                if(substr($parts['query'], 0, 2) === '__') {
                    $urlParts .= '&' . $key .'__'.$parts['filter'].$parts['query'];
                } else {
                    $urlParts .= '&' . $key .'__'.$parts['filter'].'='.$parts['query'];
                }
            }
        }
        
        $offset = 0;
        $items = [];
        $limit = $this->limit;
        $total = 0;
        $next = 'start';
 
        # Do paging
        while (1) {

            $this->logger->debug($offset);
            $response = null;
            $responseJson = null;
            
            # Check to see if the count is less offset
            if ($next == null) {
                $this->logger->debug('Get all products from revel loop finished');
                break;
            }

            $searchUrl = ($next && $next != 'start') ? $next : $endpoint . $urlParts . '&limit=' . $limit . '&offset=0';
                        
            $response = $this->client->request('GET', $searchUrl, ['headers' => ['API-AUTHENTICATION' => $this->key . ':' . $this->secret]]);

            $code = $response->getStatusCode(); // 200
            $reason = $response->getReasonPhrase(); // OK

            if ($code != 200) {
                $this->logger->error('ERROR status code ' . $code . ' Reason: ' . $reason);
                continue;
            }

            $responseJson = json_decode($response->getBody());

            if (isset($responseJson->error_code)) {
                $this->logger->critical('ERROR status code ' . $responseJson->error_code . ' Reason: ' . $responseJson->error);
                continue;
            }

            $this->logger->debug((memory_get_peak_usage(false) / 1024 / 1024) . " MiB");
            $this->logger->debug((memory_get_peak_usage(true) / 1024 / 1024) . " MiB");

            $this->logger->debug('total_count ' . $responseJson->meta->total_count);

            $next = $responseJson->meta->next;
            $total = $responseJson->meta->total_count;
            $offset+=$limit;
            # Combine results
            $items = array_merge($items, (array) $responseJson->objects);
        }

        return $items;
    }

}
