<?php
/**
 * Copyright Talisman Innovations Ltd. (2018). All rights reserved.
 */

namespace Quickbooks;

use Psr\Log\LoggerInterface;
use QuickBooksOnline\API\Core\CoreConstants;
use QuickBooksOnline\API\DataService\DataService;
use QuickBooksOnline\API\Exception\SdkException;
use QuickBooksOnline\API\Exception\ServiceException;
use ConnectorSupport\AccessToken\AccessTokenInterface;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Client;
use ConnectorSupport\Guzzle\Handler\Logger;
use ConnectorSupport\Guzzle\Handler\OAuth2;
use Talisman\TideBundle\Service\TenantLockFactory;
use Quickbooks\ApiException;

class ClientBuilder
{
    const LOCK_NAME = 'QUICKBOOKS_REFRESH_TOKEN';

    /**
     * @param string $clientId
     * @param string $clientSecret
     * @param string $environment ('development' or 'production')
     * @param string $realmId (see https://developer.intuit.com/app/developer/qbo/docs/develop/rest-api-features#base-urls)
     * @param AccessTokenInterface $token
     * @param null|LoggerInterface $logger
     * @return DataService
     * @throws ApiException
     */
    public static function getClient($clientId, $clientSecret, $environment, $realmId, AccessTokenInterface $token, TenantLockFactory $lockFactory, LoggerInterface $logger = null)
    {
        $dataService = null;

        try {

            $accessToken = $token->getAccessToken();
            $refreshToken = $token->getRefreshToken();

            $lock = $lockFactory->createLock(self::LOCK_NAME);

            $dataService = DataService::Configure(array(
                'auth_mode' => 'oauth2',
                'ClientID' => $clientId,
                'ClientSecret' => $clientSecret,
                'accessTokenKey' => $accessToken,
                'refreshTokenKey' => $refreshToken,
                'QBORealmID' => $realmId,
                'baseUrl' => $environment
            ));

            $oauthStack = HandlerStack::create();

            $oauthLog = Logger::create($logger);
            $oauthStack->push($oauthLog);

            $provider = new QuickbooksProvider([
                'clientId'              => $clientId,
                'clientSecret'          => $clientSecret,
                'redirectUri'           => '',
                'handler'               => $oauthStack
            ]);

            $clientStack = HandlerStack::create();

            $clientOauth = OAuth2::create($provider, $token, $lock, $logger);
            $clientStack->push($clientOauth);

            $clientLog = Logger::create($logger);
            $clientStack->push($clientLog);

            $guzzleClient = new Client([

                'handler' => $clientStack

            ]);

            $dataService->setClientName(CoreConstants::CLIENT_GUZZLE, $guzzleClient);

        } catch (SdkException $exception) {

            throw new ApiException(
                $exception->getMessage(),
                $exception->getCode(),
                null,
                null
            );
        }

        return $dataService;
    }

    /**
     * @param string $clientId
     * @param string $clientSecret
     * @param string $redirectUri
     * @param string $scope (see https://developer.intuit.com/app/developer/qbo/docs/develop/authentication-and-authorization/oauth-2.0#initiating-the-authorization-request)
     * @param string $environment ('development' or 'production')
     * @return string Authorization Url
     * @throws ApiException
     */
    public static function getAuthorizationUrl($clientId, $clientSecret, $redirectUri, $scope, $environment)
    {
        $url = null;

        try {

            $dataService = DataService::Configure(array(
                'auth_mode' => 'oauth2',
                'ClientID' => $clientId,
                'ClientSecret' => $clientSecret,
                'RedirectURI' => $redirectUri,
                'scope' => $scope,
                'baseUrl' => $environment
            ));

            $OAuth2LoginHelper = $dataService->getOAuth2LoginHelper();

            $url = $OAuth2LoginHelper->getAuthorizationCodeURL();

        } catch (SdkException $exception) {

            throw new ApiException(
                $exception->getMessage(),
                $exception->getCode(),
                null,
                null
            );

        }

        return $url;
    }

    /**
     * @param string $clientId
     * @param string $clientSecret
     * @param string $redirectUri
     * @param string $scope (see https://developer.intuit.com/app/developer/qbo/docs/develop/authentication-and-authorization/oauth-2.0#initiating-the-authorization-request)
     * @param string $environment ('development' or 'production')
     * @param string $authCode
     * @param string $realmId (see https://developer.intuit.com/app/developer/qbo/docs/develop/rest-api-features#base-urls)
     * @param AccessTokenInterface $token
     * @param LoggerInterface $logger
     * @throws ApiException
     */
    public static function getAccessToken($clientId, $clientSecret, $redirectUri, $scope, $environment, $authCode, $realmId, AccessTokenInterface $token, LoggerInterface $logger = null)
    {
        try {

            $dataService = DataService::Configure(array(
                'auth_mode' => 'oauth2',
                'ClientID' => $clientId,
                'ClientSecret' => $clientSecret,
                'RedirectURI' => $redirectUri,
                'scope' => $scope,
                'baseUrl' => $environment
            ));

            $clientStack = HandlerStack::create();

            $clientLog = Logger::create($logger);
            $clientStack->push($clientLog);

            $guzzleClient = new Client([

                'handler' => $clientStack

            ]);

            $dataService->setClientName(CoreConstants::CLIENT_GUZZLE, $guzzleClient);

            $OAuth2LoginHelper = $dataService->getOAuth2LoginHelper();

            $accessToken = $OAuth2LoginHelper->exchangeAuthorizationCodeForToken($authCode, $realmId);
            $dataService->updateOAuth2Token($accessToken);

            $date = $accessToken->getAccessTokenExpiresAt();
            $timestamp = strtotime($date);

            $temp = array(
                'access_token' => $accessToken->getAccessToken(),
                'refresh_token' => $accessToken->getRefreshToken(),
                'expires' => $timestamp
            );

            $token->set($temp);

        } catch (SdkException $exception) {

            throw new ApiException(
                $exception->getMessage(),
                $exception->getCode(),
                null,
                null
            );

        } catch (ServiceException $exception) {

            throw new ApiException(
                $exception->getMessage(),
                $exception->getCode(),
                null,
                null
            );

        }
    }
}