<?php
/**
 * Copyright Talisman Innovations Ltd. (2020). All rights reserved.
 */

namespace Opsuite;


use SoapClient;
use SoapFault;
use SoapHeader;
use SoapVar;

/**
 * Class WSSoapClient
 *
 * Implement Web Services Security (WSS) for SoapClient
 *
 */
class WSSSoapClient extends SoapClient
{
    private static string $OASIS = 'http://docs.oasis-open.org/wss/2004/01';

    const WSS_PASSWORD_TEXT = 1;
    const WSS_PASSWORD_DIGEST = 2;

    /**
     * WS-Security Username
     * @var string
     */
    private string $username;

    /**
     * WS-Security Password
     * @var string
     */
    private string $password;

    /**
     * WS-Security PasswordType
     * @var int
     */
    private int $passwordType;

    /**
     * WSSoapClient constructor.
     * @param $wsdl
     * @param array|null $options
     * @throws SoapFault
     */
    public function __construct($wsdl, array $options = null)
    {
        if (!isset($options['wss_username'], $options['wss_password'], $options['wss_password_type'])) {
            throw new SoapFault(100, 'Incorrect WSS options');
        }

        if (!in_array($options['wss_password_type'], [self::WSS_PASSWORD_TEXT, self::WSS_PASSWORD_DIGEST], true)) {
            throw new SoapFault(100, 'Unknown WSS password type');
        }

        $this->username = $options['wss_username'];
        $this->password = $options['wss_password'];
        $this->passwordType = $options['wss_password_type'];

        parent::__construct($wsdl, $options);
    }

    /**
     * @inheritDoc
     */
    public function __soapCall($name, $args, $options = null, $inputHeaders = null, &$outputHeaders = null): mixed
    {
        $inputHeaders[] = $this->generateWSSecurityHeader();
        return parent::__soapCall($name, $args, $options, $inputHeaders, $outputHeaders);
    }


    /**
     * Generate password digest.
     *
     * Using the password directly may work also, but it's not secure to transmit it without encryption.
     * And anyway, at least with axis+wss4j, the nonce and timestamp are mandatory anyway.
     *
     * @param int $nonce
     * @param string $timestamp
     * @return string   base64 encoded password digest
     */
    private function generatePasswordDigest(int $nonce, string $timestamp): string
    {
        $packedNonce = pack('H*', $nonce);
        $packedTimestamp = pack('a*', $timestamp);
        $packedPassword = pack('a*', $this->password);

        $hash = sha1($packedNonce . $packedTimestamp . $packedPassword);
        $packedHash = pack('H*', $hash);

        return base64_encode($packedHash);
    }

    /**
     * Generates WS-Security headers
     *
     * @return SoapHeader|null
     */
    private function generateWSSecurityHeader(): ?SoapHeader
    {
        $timestamp = gmdate('Y-m-d\TH:i:s\Z');

        switch ($this->passwordType) {
            case self::WSS_PASSWORD_DIGEST:
                $rand = mt_rand();
                $password = $this->generatePasswordDigest($rand, $timestamp);
                $passwordType = 'PasswordDigest';
                $nonce = sha1($rand);
                break;
            case self::WSS_PASSWORD_TEXT:
                $password = htmlspecialchars($this->password, ENT_XML1);
                $passwordType = 'PasswordText';
                $nonce = sha1(mt_rand());
                break;
            default:
                return null;
        }

        $xml =
'<wsse:Security SOAP-ENV:mustUnderstand="1" xmlns:wsse="' . self::$OASIS . '/oasis-200401-wss-wssecurity-secext-1.0.xsd">
	<wsse:UsernameToken>
	<wsse:Username>' . htmlspecialchars($this->username, ENT_XML1)  . '</wsse:Username>
	<wsse:Password Type="' . self::$OASIS . '/oasis-200401-wss-username-token-profile-1.0#' . $passwordType . '">' . $password . '</wsse:Password>
	<wsse:Nonce EncodingType="' . self::$OASIS . '/oasis-200401-wss-soap-message-security-1.0#Base64Binary">' . $nonce . '</wsse:Nonce>
	<wsu:Created xmlns:wsu="' . self::$OASIS . '/oasis-200401-wss-wssecurity-utility-1.0.xsd">' . $timestamp . '</wsu:Created>
	</wsse:UsernameToken>
</wsse:Security>';

        return new SoapHeader(self::$OASIS . '/oasis-200401-wss-wssecurity-secext-1.0.xsd',
            'Security',
            new SoapVar($xml, XSD_ANYXML),true);
    }
}