<?php

declare(strict_types=1);

/*
 * Copyright Talisman Innovations Ltd. (2016). All rights reserved.
 *
 */

namespace Lightspeed;

use Exception;
use Psr\Log\LoggerInterface;

class Remote
{
    protected LoggerInterface $logger;
    protected Logger $curlLogger;
    protected string $token;
    protected Auth $auth;

    private const string BASE_URL = 'https://api.lightspeedapp.com/API/V3/Account/';

    public function __construct(LoggerInterface $logger, string $token, Auth $auth)
    {
        $this->logger = $logger;
        $this->curlLogger = new Logger($logger);
        $this->token = $token;
        $this->auth = $auth;
    }

    public function get(?string $endpoint = null): string
    {
        return $this->call('get', $endpoint);
    }

    public function put(string $endpoint, array|object $data): string
    {
        return $this->call('put', $endpoint, $data);
    }

    public function post(string $endpoint, array|object $data): string
    {
        return $this->call('post', $endpoint, $data);
    }

    public function delete(string $endpoint): string
    {
        return $this->call('delete', $endpoint);
    }

    private function call(string $callType, ?string $endpoint, array|object|null $data = null): string
    {
        $callType = strtoupper($callType);

        [$info, $body, $headers] = $this->curl($callType, $endpoint, $data);

        return match ($info['http_code']) {
            429 => $this->handleRateLimit($callType, $headers, $endpoint, $data),
            401 => $this->handleUnauthorized($callType, $endpoint, $data),
            200 => $body,
            default => throw new Exception($this->getErrorMessage($info['http_code']))
        };
    }

    private function handleRateLimit(string $callType, array $headers, ?string $endpoint, array|object|null $data): string
    {
        $this->leakyBucket($headers);
        return $this->call($callType, $endpoint, $data);
    }

    private function handleUnauthorized(string $callType, ?string $endpoint, array|object|null $data): string
    {
        $this->tokenReset();
        return $this->call($callType, $endpoint, $data);
    }

    private function curl(string $callType, ?string $endpoint, array|object|null $data): array
    {
        $curl = curl_init();
        $url = self::BASE_URL . $endpoint;
        $body = $data ? json_encode($data) : null;

        // Base curl setup for all calls
        curl_setopt($curl, CURLOPT_URL, $url);
        curl_setopt($curl, CURLOPT_ACCEPT_ENCODING, 'application/json');
        curl_setopt($curl, CURLOPT_FRESH_CONNECT, true);
        curl_setopt($curl, CURLINFO_HEADER_OUT, true);
        curl_setopt($curl, CURLOPT_HEADER, true);
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($curl, CURLOPT_TIMEOUT, 120);
        curl_setopt($curl, CURLOPT_CONNECTTIMEOUT, 20);
        // these options allow us to read the error message sent by the API
        curl_setopt($curl, CURLOPT_FAILONERROR, false);
        curl_setopt($curl, CURLOPT_HTTP200ALIASES, range(400, 599));

        $reqHeaders = [
            "Authorization: Bearer $this->token",
        ];

        switch ($callType) {
            case 'PUT':
            case 'POST':
                $reqHeaders[] = 'Content-Type: application/json';
                curl_setopt($curl, CURLOPT_CUSTOMREQUEST, $callType);
                curl_setopt($curl, CURLOPT_POSTFIELDS, $body);
                break;
            case 'DELETE':
                curl_setopt($curl, CURLOPT_CUSTOMREQUEST, 'DELETE');
                break;
        }

        curl_setopt($curl, CURLOPT_HTTPHEADER, $reqHeaders);

        $respHeaders = [];
        curl_setopt($curl, CURLOPT_HEADERFUNCTION,
            function ($curl, $header) use (&$respHeaders) {
                $len = strlen($header);
                $header = explode(':', $header, 2);
                if (count($header) < 2) // ignore invalid headers
                    return $len;

                $respHeaders[strtolower(trim($header[0]))][] = trim($header[1]);

                return $len;
            }
        );

        $response = curl_exec($curl);
        $info = curl_getinfo($curl);
        $response = substr($response, $info['header_size']);

        $this->curlLogger->log($callType, $url, '', $reqHeaders, $body, $info['http_code'], $respHeaders, $response);

        curl_close($curl);

        return [$info, $response, $respHeaders];
    }

    private function leakyBucket(array $headers): void
    {
        $sleep = 1;
        foreach ($headers as $key => $value) {
            if (strtolower($key) === 'retry-after') {
                $sleep = (int) $value[0];
                break;
            }
        }

        $this->logger->warning(sprintf('429 received, sleep for %d seconds', $sleep));
        usleep($sleep * 1000000); // Convert seconds to microseconds
    }

    private function getErrorMessage(int $code): string
    {
        return match ($code) {
            400 => 'Bad Request: The server cannot or will not process the request due to something that is perceived to be a client error (ie. malformed query or invalid XML/JSON payload)',
            401 => 'Auth token expired reset it',
            403 => 'Forbidden: The request was a valid request, but the server is refusing to respond to it. Unlike a 401 Unauthorized response, authenticating will make no difference',
            404 => 'Not Found: The requested resource could not be found but may be available again in the future',
            405 => 'Method Not Allowed: A request was made of a resource using a request method not supported by that resource; for example, using GET on a form which requires data to be presented via POST, or using PUT on a read-only resource',
            409 => 'Conflict: Indicates that the request could not be processed because of conflict in the request, such as an edit conflict in the case of multiple updates',
            422 => 'Unprocessable Entity: The request was well-formed but was unable to be followed due to semantic errors',
            429 => 'Too Many Requests: The user has sent too many requests in a given amount of time. Intended for use with rate limiting schemes',
            500 => 'Internal Error: A generic error message, given when an unexpected condition was encountered and no more specific message is suitable',
            502 => 'Bad Gateway: The server was acting as a gateway or proxy and received an invalid response from the upstream server',
            503 => 'Service Unavailable: The server is currently unavailable (because it is overloaded or down for maintenance)',
            default => "Unknown HTTP error code: {$code}"
        };
    }

    protected function tokenReset(): void
    {
        $this->auth->refreshToken();
        $this->logger->debug('401 received resetting tokens');
        $this->token = $this->auth->getAccessToken();
    }

}