<?php

declare(strict_types=1);

/*
 * Copyright Talisman Innovations Ltd. (2016). All rights reserved.
 */

namespace Lightspeed;

use DateTime;
use DateTimeZone;
use Psr\Log\LoggerInterface;

abstract class CallAbstract {

    private const int LIMIT = 100;

    protected Remote $remote;
    protected ?string $account;
    protected string $endpoint;
    protected LoggerInterface $logger;
    protected string $token;
    protected string $timezone;
    protected string $format;
    protected Auth $auth;

      public function __construct(LoggerInterface $logger, string $token, Auth $auth, ?string $accountId) {
        $this->remote = new Remote($logger, $token, $auth);
        $this->account = $accountId;
        $this->logger = $logger;
        $this->token = $token;
        $this->timezone = 'UTC';
        $this->format = 'json';
        $this->auth = $auth;
    }

    protected function getAccountId(): string {
        if($this->account) {
            return $this->account;
        }
        $this->logger->debug('No Account Id - resetting.');
        $account = json_decode($this->remote->get());
        $this->account = $account->Account->accountID;
        return $account->Account->accountID;
    }

    protected function getSingle(int $id, ?string $pathPart = null): object {
        $pathPart = $pathPart ? '/' . $pathPart : '';
        $result = $this->remote->get($this->getAccountId() . '/' . $this->endpoint . '/' . $id . $pathPart . '.' . $this->format);

        return json_decode($result);
    }

    protected function getAll(?array $query = null): object {
        $queryString = '';
        if ($query) {
            $queryParts = [];
            foreach ($query as $key => $value) {
                $operator = $value['operator'] ? urlencode($value['operator'] . ',') : '';
                $queryParts[] = $key . '=' . $operator . $value['value'];
            }
            $queryString = '&' . implode('&', $queryParts);
        }

        $next = null;
        $items = [];
        $endpoint = $this->endpoint;

        while (true) {
            $call = $next ?: $this->getAccountId() . '/' . $this->endpoint . '.' . $this->format . '?limit=' . self::LIMIT . $queryString;

            $result = json_decode($this->remote->get($call));
            if (!$result) {
                $this->logger->info('Null result received for call ' . $call);
                break;
            }

            $setData = is_array($result->{$endpoint}) ? $result->{$endpoint} : [$result->{$endpoint}];
            $items[] = $setData;

            $count = count($setData);
            if ($count <= 1 || $count < self::LIMIT) {
                break;
            }

            $attributes = $result->{'@attributes'};
            $next = $attributes->next ?? null;
            if (!$next) {
                break;
            }
        }

        return (object) [$endpoint => array_merge(...$items)];
    }

    protected function create(array|object $data): object {
        $result = $this->remote->post($this->getAccountId() . '/' . $this->endpoint, $data);
        return json_decode($result);
    }

    protected function update(int $id, array|object $data): object {
        $result = $this->remote->put($this->getAccountId() . '/' . $this->endpoint . '/' . $id, $data);
        return json_decode($result);
    }

    protected function delete(int $id): object {
        $result = $this->remote->delete($this->getAccountId() . '/' . $this->endpoint . '/' . $id);
        return json_decode($result);
    }

    protected function search(?array $query): ?object {
        if (!$query) {
            return null;
        }
        return $this->getAll($query);
    }

    protected function getBetweenDates(string $fromDate, string $toDate): ?object {
        $querystring = 'timeStamp >= ' . $this->formatDate($fromDate) . '&timeStamp <= ' . $this->formatDate($toDate);
        return $this->search([$querystring]);
    }

    protected function getAfter(string $date): ?object {
        $querystring = 'timeStamp > ' . $this->formatDate($date);
        return $this->search([$querystring]);
    }

    protected function getBefore(string $date): ?object {
        $querystring = 'timeStamp < ' . $this->formatDate($date);
        return $this->search([$querystring]);
    }

    protected function formatDate(string $date): string {
        $setDate = new DateTime($date, new DateTimeZone($this->timezone));
        return $setDate->format('Y-m-d\TH:i:s.u');
    }

    public function getAccessToken(): string {
        return $this->auth->getAccessToken();
    }

}
