<?php

/*
 * Copyright Talisman Innovations Ltd. (2016). All rights reserved.
 * 
 */

namespace Lightspeed;

use Exception;
use Psr\Log\LoggerInterface;

class Remote
{
    /** @var LoggerInterface */
    protected $logger;
    /** @var Logger */
    protected $curlLogger;
    protected $token;
    protected $auth;

    const BASE_URL = 'https://api.lightspeedapp.com/API/V3/Account/';

    public function __construct(LoggerInterface $logger, $token, $auth)
    {
        $this->logger = $logger;
        $this->curlLogger = new Logger($logger);
        $this->token = $token;
        $this->auth = $auth;
    }

    /**
     * @param null $endpoint
     * @return string
     * @throws Exception
     */
    public function get($endpoint = null)
    {
        return $this->call('get', $endpoint);
    }

    /**
     * @param $endpoint
     * @param $data
     * @return string
     * @throws Exception
     */
    public function put($endpoint, $data)
    {
        return $this->call('put', $endpoint, $data);
    }

    /**
     * @param $endpoint
     * @param $data
     * @return string
     * @throws Exception
     */
    public function post($endpoint, $data)
    {
        return $this->call('post', $endpoint, $data);
    }

    /**
     * @param $endpoint
     * @return string
     * @throws Exception
     */
    public function delete($endpoint)
    {
        return $this->call('delete', $endpoint);
    }

    /**
     * Use get to return data from lightspeed
     * @param string $callType
     * @param string $endpoint
     * @param null $data
     * @return string
     * @throws Exception
     */
    private function call(string $callType, string $endpoint, $data = null)
    {
        $callType = strtoupper($callType);

        [$info, $body, $headers] = $this->curl($callType, $endpoint, $data);

        # Check for a timeout http code, recall if necessary
        switch ($info['http_code']) {
            case 429:
                $this->leakyBucket($callType, $headers);
                $body = $this->call($callType, $endpoint, $data);
                break;
            case 401:
                $this->tokenReset();
                $body = $this->call($callType, $endpoint, $data);
                break;
            case 200:
                return $body;
            default :
                $this->errorCodes($info['http_code']);
        }
        return $body;
    }

    /**
     * @param string $callType
     * @param string $endpoint
     * @param mixed $data
     * @return array
     */
    private function curl(string $callType, string $endpoint, $data)
    {
        $curl = curl_init();
        $url = self::BASE_URL . $endpoint;
        $body = $data ? json_encode($data) : null;

        // Base curl setup for all calls
        curl_setopt($curl, CURLOPT_URL, $url);
        curl_setopt($curl, CURLOPT_FRESH_CONNECT, true);
        curl_setopt($curl, CURLINFO_HEADER_OUT, true);
        curl_setopt($curl, CURLOPT_HEADER, true);
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($curl, CURLOPT_TIMEOUT, 120);
        curl_setopt($curl, CURLOPT_CONNECTTIMEOUT, 20);
        // these options allow us to read the error message sent by the API
        curl_setopt($curl, CURLOPT_FAILONERROR, false);
        curl_setopt($curl, CURLOPT_HTTP200ALIASES, range(400, 599));

        $reqHeaders = [
            'Accept: application/json',
            "Authorization: OAuth $this->token",
            "Host: api.merchantos.com",
        ];

        switch ($callType) {
            case 'PUT':
            case 'POST':
                $reqHeaders[] = 'Content-Type: application/json';
                curl_setopt($curl, CURLOPT_CUSTOMREQUEST, $callType);
                curl_setopt($curl, CURLOPT_POSTFIELDS, $body);
                break;
            case 'DELETE':
                curl_setopt($curl, CURLOPT_CUSTOMREQUEST, 'DELETE');
                break;
        }

        curl_setopt($curl, CURLOPT_HTTPHEADER, $reqHeaders);

        $respHeaders = [];
        curl_setopt($curl, CURLOPT_HEADERFUNCTION,
            function ($curl, $header) use (&$respHeaders) {
                $len = strlen($header);
                $header = explode(':', $header, 2);
                if (count($header) < 2) // ignore invalid headers
                    return $len;

                $respHeaders[strtolower(trim($header[0]))][] = trim($header[1]);

                return $len;
            }
        );

        $response = curl_exec($curl);
        $info = curl_getinfo($curl);
        $response = substr($response, $info['header_size']);

        $this->curlLogger->log($callType, $url, '', $reqHeaders, $body, $info['http_code'], $respHeaders, $response);

        curl_close($curl);

        return [$info, $response, $respHeaders];
    }

    /**
     * Workout from the header
     * the lowest sleep time
     * @param $callType
     * @param array $headers
     */
    private function leakyBucket($callType, array $headers)
    {
        $sleep = 1;
        foreach ($headers as $key => $value) {
            if (strtolower($key) == 'retry-after') {
                $sleep = $value[0];
            }
        }

        $this->logger->warning(sprintf('429 received, sleep for %d seconds', $sleep));
        usleep($sleep);
    }

    /**
     * Check for http errors
     * @param int $code
     * @throws Exception
     */
    private function errorCodes(int $code)
    {
        $error = null;

        switch ($code) {
            case 400:
                $error = 'Bad Request: The server cannot or will not process the request due to something that is perceived to be a client error (ie. malformed query or invalid XML/JSON payload)';
                break;
            case 401:
                $error = '# Auth token expired reset it';
                break;
            case 403:
                $error = 'Forbidden: The request was a valid request, but the server is refusing to respond to it. Unlike a 401 Unauthorized response, authenticating will make no difference';
                break;
            case 404:
                $error = 'Not Found: The requested resource could not be found but may be available again in the future';
                break;
            case 405:
                $error = 'Method Not Allowed: A request was made of a resource using a request method not supported by that resource; for example, using GET on a form which requires data to be presented via POST, or using PUT on a read-only resource';
                break;
            case 409:
                $error = 'Conflict: Indicates that the request could not be processed because of conflict in the request, such as an edit conflict in the case of multiple updates';
                break;
            case 422:
                $error = 'Unprocessable Entity: The request was well-formed but was unable to be followed due to semantic errors';
                break;
            case 429:
                $error = 'Too Many Requests: The user has sent too many requests in a given amount of time. Intended for use with rate limiting schemes';
                break;
            case 500:
                $error = 'Internal Error: A generic error message, given when an unexpected condition was encountered and no more specific message is suitable';
                break;
            case 502:
                $error = 'Bad Gateway: The server was acting as a gateway or proxy and received an invalid response from the upstream server';
                break;
            case 503:
                $error = 'Service Unavailable: The server is currently unavailable (because it is overloaded or down for maintenance)';
                break;
        }

        if ($error) {
            $this->logger->critical($error);
            throw new Exception($error);
        }
    }

    /**
     * if 401 received reset the LS token
     */
    protected function tokenReset()
    {
        $this->auth->refreshToken();
        $this->logger->debug('401 received resetting tokens');
        $this->token = $this->auth->getAccessToken();
    }

}