<?php

/*
 * Copyright Talisman Innovations Ltd. (2016). All rights reserved.
 */

namespace Lightspeed;

use DateTime;
use DateTimeZone;
use Psr\Log\LoggerInterface;

abstract class CallAbstract {

    private const LIMIT = 100;

    protected $remote;
    protected $account;
    protected $endpoint;
    protected $logger;
    protected $token;
    protected $timezone;
    protected $format;
    protected $auth;

      public function __construct(LoggerInterface $logger, $token, $auth, $accountId) {
        $this->remote = new Remote($logger, $token, $auth);
        $this->account = $accountId;
        $this->logger = $logger;
        $this->timezone = 'UTC';
        $this->format = 'json';
        $this->auth = $auth;
    }

    /**
     * Get the account ID used in all endpoints
     * @return string
     * @throws \Exception
     */
    protected function getAccountId() {
        if($this->account) {
            return $this->account;
        }
        $this->logger->debug('No Account Id - resetting.');
        $account = json_decode($this->remote->get());
        $this->account = $this->account->Account->accountID;
        return $this->account->Account->accountID;
    }

    /**
     * Get a single record
     * @param int $id
     * @return object
     * @throws \Exception
     */
    protected function getSingle($id, $pathPart = null) {

        if (is_null($pathPart)) {
            $pathPart = '';
        } else {
            $pathPart = '/' . $pathPart;
        }
        
        $result = $this->remote->get($this->getAccountId() . '/' . $this->endpoint . '/' . $id . $pathPart . '.' . $this->format);
        
        return json_decode($result);
    }

    /**
     * Get All records
     * @return object
     * @throws \Exception
     * @var $query array
     */
    protected function getAll($query = null) {

        $queryString = '';
        if ($query) {
            foreach ($query as $key => $value) {
                $join = (end($query)['value'] != $value['value']) ? '&' : '';
                $operator = ($value['operator']) ? urlencode($value['operator'] . ',') : '';
                $queryString .= $key . '=' . $operator . '' . $value['value'] . '' . $join;
            }
            $queryString = '&' . $queryString;
        }

        $next = null;
        $items = [];
        $endpoint = (string) $this->endpoint;

        while (1) {
            if($next) {
                $call = $next;
            } else {
                $call = $this->getAccountId() . '/' . $this->endpoint . '.' . $this->format . '?limit='.self::LIMIT .  $queryString;
            }

            $result = json_decode($this->remote->get($call));
            if(!$result) {
                $this->logger->info('Null result received for call '.$call);
                break;
            }

            # If a single object is returned not an array make
            # it into an array, so we can merge with the others
            $setData = is_array($result->{$endpoint}) ? $result->{$endpoint} : [$result->{$endpoint}];
            $items[] = $setData;

            $count = count($setData);
            # Break out for a single result
            if ($count <= 1 || $count < self::LIMIT) {
                break;
            }



            # Get the attributes block, next and previous
            # check for next leave it loose as can be "" or null
            # break out
            $attributes = $result->{'@attributes'};
            $next = $attributes->next;
            if(!$next) {
                break;
            }
        }

        return (object) [$endpoint => array_merge(...$items)];
    }

    /**
     * Create a new record
     * @param array or object $data
     * @return object
     * @throws \Exception
     */
    protected function create($data) {
        $result = $this->remote->post($this->getAccountId() . '/' . $this->endpoint, $data);
        return json_decode($result);
    }

    /**
     * Update a record
     * @param int $id
     * @param array or object $data
     * @return object
     * @throws \Exception
     */
    protected function update($id, $data) {
        $result = $this->remote->put($this->getAccountId() . '/' . $this->endpoint . '/' . $id, $data);
        return json_decode($result);
    }

    /**
     * Delete a record
     * (not delete for some record types i.e. Customer will Archive the record
     * @param int $id
     * @return object
     * @throws \Exception
     */
    protected function delete($id) {
        $result = $this->remote->delete($this->getAccountId() . '/' . $this->endpoint . '/' . $id);
        return json_decode($result); 
    }

    /**
     * Pass in a quert to an endpoint
     * Example usage
     * $query = [];
     * $query['createTime'] = ['operator'=>'>=','value'=>$date->format('Y-m-d\TH:i:s')];
     * $query['registerCountID'] = ['operator'=>'','value'=>1];
     * ->search($query)
     * @param array $query
     * @return object
     */
    protected function search($query) {
        if (!$query) {
            return;
        }
        return $this->getAll($query);
    }

    protected function getBetweenDates($fromDate, $toDate) {
        $querystring = 'timeStamp >= ' . $this->formatDate($fromDate) . '&timeStamp <= ' . $this->formatDate($toDate);
        return $this->search($querystring);
    }

    protected function getAfter($date) {
        $querystring = 'timeStamp > ' . $this->formatDate($date);
        return $this->search($querystring);
    }

    protected function getBefore($date) {
        $querystring = 'timeStamp < ' . $this->formatDate($date);
        return $this->search($querystring);
    }

    protected function formatDate($date) {
        $setDate = new DateTime($date, new DateTimeZone($this->timezone));
        return $setDate->format('Y-m-dTH:i:s.u');
    }

    public function getAccessToken() {
        return $this->auth->getAccessToken();
    }

}
