<?php

/*
 * Copyright Talisman Innovations Ltd. (2016). All rights reserved.
 * 
 * Lightspeed OAuth
 * 
 */

namespace Lightspeed;

use Exception;
use Psr\Log\LoggerInterface;

class Auth {

    private $authUri;
    public $accessToken;
    public $refreshToken;
    public $tokenType;
    public $expires;
    private $clientId;
    private $clientSecret;
    private $scope;
    private $logger;
    private $returnUrl;

    public function __construct(LoggerInterface $logger, $clientId, $clientSecret, $scope, $format, $returnUrl, $token=null, $refreshToken=null) {
        
        $this->authUri = 'https://cloud.merchantos.com/oauth/';
        $this->returnUrl = $returnUrl;
        $this->clientId = $clientId;
        $this->clientSecret = $clientSecret;
        if($token) {
           $this->accessToken = $token; 
        }
        if($refreshToken) {
           $this->refreshToken = $refreshToken; 
        }
        $this->scope = $scope;
        $this->logger = $logger;
    }

    /**
     * Redirect to Lightspeed to get an
     * authorisation code
     * Will return to a given url 
     * @throws Exception
     */
    public function auth() {

        if (!$this->scope || !$this->clientId) {
            $this->logger->critical('Client ID and Scope required for auth redirect', [$this]);
            throw new Exception('Client ID and Scope required for auth redirect');
        }
        $authURL = $this->authUri . "authorize.php?response_type=code&client_id={$this->clientId}&scope={$this->scope}";
        header("location: {$authURL}");

        return;
    }

    /**
     * Using the returned $code (GET)
     * Call lightspeed and get the Tokens etc
     * @param string $code
     * @return boolean
     * @throws Exception
     */
    public function setTokens($code) {

        if (!$code) {
            $this->logger->critical('Code returned from Lightspeed required', [$this]);
            throw new Exception('Code returned from Lightspeed required');
        }

        $postFields = [
            'client_id' => $this->clientId,
            'client_secret' => $this->clientSecret,
            'code' => $code,
            'grant_type' => 'authorization_code',
            'redirect_uri' => $this->returnUrl
        ];

        $result = json_decode($this->call($postFields));
       
        if(!isset($result->access_token)) {
            $this->logger->critical('Access tokens not returned see error:'.serialize($result));
            throw new Exception('Access tokens not returned see error:'.serialize($result));
        }

        $this->accessToken = $result->access_token;
        $this->refreshToken = $result->refresh_token;
        $this->tokenType = $result->token_type;
        $this->expires = $result->expires_in;

        return true;
    }

    /**
     * If a token is out of date use the 
     * refresh token to get a new one
     * @return boolean
     */
    public function refreshToken() {

        $postFields = [
            'client_id' => $this->clientId,
            'client_secret' => $this->clientSecret,
            'grant_type' => 'refresh_token',
            'refresh_token' => $this->refreshToken
        ];

        $result = json_decode($this->call($postFields));
        
        $this->accessToken = $result->access_token;
        $this->tokenType = $result->token_type;
        $this->expires = $result->expires_in;

        return true;
    }

    /**
     * Get access token
     * Store it where you like
     * @return string
     */
    public function getAccessToken() {
        return $this->accessToken;
    }
    
    /**
     * Get refresh token
     * Not currently used
     * Store it where you like
     * @return string
     */
    public function getRefreshToken() {
        return $this->refreshToken;
    }
    
    /**
     * Get token type i.e. bearer
     * Store it where you like
     * @return string
     */
    public function getTokenType() {
        return $this->tokenType;
    }

    /**
     * Get expires
     * Not currently used 
     * Store it where you like
     * @return string
     */
    public function getExpires() {
        return $this->expires;
    }
    
    /**
     * Get a array listing all available scopes
     * that can be used when authorising the App
     * @return array
     */
    public function getScopes() {
        
        $scopes = [];
        $scopes[] = 'employee:all'; # Grants full read and write access to the account
        $scopes[] = 'employee:register'; # Create new Sales and read sales history.
        $scopes[] = 'employee:inventory'; # View, create, update, and archive items and inventory
        $scopes[] = 'employee:reports'; # View reports
        $scopes[] = 'employee:admin'; #	View, create, update, and archive administrative records
        $scopes[] = 'employee:admin_shops'; #	View, create, update, and archive shops
        $scopes[] = 'employee:admin_employees'; # View, create, update, and archive employees
        $scopes[] = 'employee:admin_purchases'; # View, create, update, and archive payment types, discounts, and taxes
        $scopes[] = 'employee:admin_void_sale'; # Void sales
        $scopes[] = 'employee:admin_inventory'; # View, create, update, and archive vendors and manufacturers
        $scopes[] = 'employee:workbench'; # View, create, update, and archive work orders
        $scopes[] = 'employee:customers'; # View, create, update, and archive customers
        $scopes[] = 'employee:customers_credit_limit'; # View, create, update, and archive customer credit accounts
        
        return $scopes;
    }

    
    /**
     * Authorisation specific curl call
     * @param array $postFields
     */
    private function call($postFields) {

        $curl = curl_init();
        curl_setopt_array($curl, array(
            CURLOPT_URL => $this->authUri . 'access_token.php',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => "",
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_CUSTOMREQUEST => "POST",
            CURLOPT_POSTFIELDS => $postFields
        ));
        $result = curl_exec($curl);
        $info = curl_getinfo($curl);
        $error = curl_error($curl);

        $this->logger->debug('CURL INFO:', $info);
        $this->logger->debug('CURL  RESULT:' . $result);
        $this->logger->debug('CURL ERROR:' . $error);
        
        return $result;
    }

}
