# Claude Code Configuration for Lightspeed Retail PHP API

This file contains configuration and notes for working with the Lightspeed Retail PHP API library using Claude Code.

## Project Structure

This PHP library was generated using OpenAPI Generator v7.16.0 from the Lightspeed Retail R-Series API specification.

### Directory Layout
- `lib/` - Main library code (Models and API classes)
- `docs/` - Auto-generated documentation
- `test/` - PHPUnit test cases
- `openapi/` - Configuration files for OpenAPI Generator
- `composer.json` - Composer dependencies and autoloader configuration

### Key Components
- **Models**: Located in `lib/Model/` - Data transfer objects for API resources
- **APIs**: Located in `lib/Api/` - API client classes for different endpoint groups
- **Configuration**: Located in `lib/` - Client configuration and authentication

## Development Commands

### Testing
```bash
# Run all tests
vendor/bin/phpunit

# Run specific test class
vendor/bin/phpunit test/Model/AccountTest.php
```

### Code Style
```bash
# Fix code style issues
vendor/bin/php-cs-fixer fix
```

### Dependencies
```bash
# Install dependencies
composer install

# Update dependencies
composer update
```

## API Usage Examples

### Basic Setup
```php
<?php
require_once(__DIR__ . '/vendor/autoload.php');

use Lightspeed\Retail\Configuration;
use Lightspeed\Retail\Api\AccountApi;

$config = Configuration::getDefaultConfiguration()
    ->setAccessToken('your_access_token_here');

$apiInstance = new AccountApi(
    new GuzzleHttp\Client(),
    $config
);
```

### Get Account Information
```php
try {
    $result = $apiInstance->accountGet();
    print_r($result);
} catch (Exception $e) {
    echo 'Exception: ', $e->getMessage(), PHP_EOL;
}
```

## Authentication

The Lightspeed Retail API uses Bearer token authentication. Set your access token in the configuration:

```php
$config = Configuration::getDefaultConfiguration()
    ->setAccessToken('your_lightspeed_access_token');
```

## Resources Covered

This library provides comprehensive coverage of the Lightspeed Retail R-Series API including:

- Account management
- Customer operations
- Item/Product management
- Sales and orders
- Inventory management
- Employee management
- Categories and pricing
- Reports and analytics

## Generated from OpenAPI Specification

- **Source**: `/openapi/openapi.yaml`
- **Generator**: OpenAPI Generator v7.16.0
- **Generator Config**: `/openapi/config.json`
- **Language**: PHP 8.1+
- **HTTP Client**: Guzzle 7.3+

## Notes

- This library requires PHP 8.1 or higher
- Uses Guzzle HTTP client for API requests
- Follows PSR-4 autoloading standards
- Includes comprehensive PHPUnit test coverage
- Documentation can be found here https://developers.lightspeedhq.com/retail/introduction/introduction/
- For example, the Account endpoint is described here https://developers.lightspeedhq.com/retail/endpoints/Account/