<?php

/*
 * Copyright Talisman Innovations Ltd. (2018). All rights reserved.
 */

namespace Talisman\EurostopBundle\Eurostop;

use ConnectorSupport\Curl\Logger;
use Psr\Log\LoggerInterface;
use Exception;
use SoapClient;

/**
 * Eurostop Base Class
 *
 */
class Base
{
    /** @var Logger  */
    protected $logger;
    protected $wsdl;
    protected $soapOptions;
    protected $endpoint;
    /** @var SoapClient */
    protected $client;

    /**
     * Base constructor.
     * @param LoggerInterface $logger
     * @param string $wsdl
     * @param $soapOptions
     */
    public function __construct(LoggerInterface $logger, $wsdl, $soapOptions) {
        $this->logger = new Logger($logger);
        $this->wsdl = $wsdl;
        $this->soapOptions = $soapOptions;
    }


    /**
     * Get service class
     */
    protected function service() {
        $dir = $this->endpoint;
        if($this->endpoint == 'stock') {
            $dir = $this->endpoint.'s';
        }
        $interfaceName = '\\Eurostop\\'.ucfirst($dir).'\\'.ucfirst($this->endpoint).'WebServiceInterfacePlugin';
        return new $interfaceName($this->soapOptions, $this->wsdl);
    }


    /**
     * Break up the response object and return the usuable
     * parts like items.
     * @param object $response
     * @return object or array of objects
     * @throws Exception
     */
    protected function response($response) {

        if (is_null($response)) {
            return $response;
        }
        $classMethods = get_class_methods($response);
        $result = $response->{$classMethods[1]}();

        if($result->getErrorSource()) {
            throw new Exception('Eurostop: '.ucfirst($this->endpoint). ' '.$result->getErrorSource());
        }
        # Single item
        if($result->getDataItem()) {
            return [$result->getDataItem()];
        }
        # Array of results
        if($result->getDataItems()) {
            return $result->getDataItems();
        }

        return null;
    }

    protected function LogFaults($fault)
    {
        $this->LogLastRequest();
        $this->logger->info(json_encode($fault));
    }

    protected function LogLastRequest()
    {
        $bodySent = $this->client->__getLastRequest();
        $headersSent = $this->client->__getLastRequestHeaders();
        $headers = explode("\r\n", $headersSent);
        $temp = explode(" ", array_shift($headers));
        $method = $temp[0];
        $url = $temp[1];
        $sent = $this->parseHeaders($headers);

        if (array_key_exists('Host', $sent)) {
            $url = $sent['Host'] . $url;
        }

        $bodyRecv = $this->client->__getLastResponse();
        $headersRecv = $this->client->__getLastResponseHeaders();
        $headers = explode("\r\n", $headersRecv);
        $temp = explode(" ", array_shift($headers));
        $http_code = $temp[1];
        $recv = $this->parseHeaders($headers);

        $this->logger->log($method, $url, null, $sent, $bodySent, $http_code, $recv, $bodyRecv);
    }

    /**
     * @param array $headers
     * @return array
     */
    protected function parseHeaders(array $headers)
    {
        $parsedHeaders = [];
        foreach ($headers as $header) {
            $temp = explode(':', $header, 2);
            if (count($temp) > 1) {
                $parsedHeaders[trim($temp[0])] = trim($temp[1]);
            }
        }

        return $parsedHeaders;
    }
}
