<?php

/* 
 * Copyright Talisman Innovations Ltd. (2018). All rights reserved.
 */

namespace Talisman\EurostopBundle\Eurostop;

use Psr\Log\LoggerInterface;
use SoapFault;
use Eurostop\Products\GetProductsModifiedSinceDate;
use Eurostop\Products\GetSkuPricesByCountry;
use Eurostop\Products\GetSkuPricesByCountryAndProduct;
use Eurostop\Products\GetColour;
use Eurostop\Products\GetSize;
use Talisman\EurostopBundle\Eurostop\Base;
use Eurostop\Products\GetAllColours;
use Eurostop\Products\GetAllProductAnalysisValues;
use Eurostop\Products\GetAllHierarchyNodes;
use Eurostop\Products\GetAllProductAnalysisTypes;
use Eurostop\Products\GetAllExtendedProductAttributeTypes;
use Eurostop\Products\GetAllSizeGroups;
use Eurostop\Products\GetAllSizes;
use Eurostop\Products\GetAllSkuAnalysisTypes;
use Eurostop\Products\GetAllSkuAnalysisValues;
use Eurostop\Products\GetAllSuppliers;
use Eurostop\Products\GetSkuPriceByCountryAndSku;
use Eurostop\Products\GetProduct;
use Eurostop\Products\GetSkuPricesBySku;
use DateTime;

/**
 * Eurostop Product Class 
 * 
 */

class Products extends Base {
    
    protected $endpoint = 'products';
    protected $client;

    public function __construct(LoggerInterface $logger, $url, $soapOptions) {
        parent::__construct($logger, $url. $this->endpoint . '?wsdl', $soapOptions);
        $this->client = $this->service();
    }
    
    


    ######################### PRODUCTS #########################
    
    /**
     * Get Products modified since
     * @param DateTime $modifiedSinceDate
     * @param type $detectProductsModifiedSinceDate
     * @param bool $detectSkusModifiedSinceDate
     * @param bool $returnAnalysisValues
     * @param bool $returnExtendedAttributeValues
     * @param string $cultureCode
     * @param DateTime $modifiedUntilDate
     * @return type
     */
    public function getProductsModifiedSinceDate(DateTime $modifiedSinceDate, $detectProductsModifiedSinceDate, $detectSkusModifiedSinceDate, $returnAnalysisValues = true, $returnExtendedAttributeValues = false, $cultureCode=null, DateTime $modifiedUntilDate=null) {
        $request = new GetProductsModifiedSinceDate($modifiedSinceDate, $detectProductsModifiedSinceDate, $detectSkusModifiedSinceDate, $returnAnalysisValues, $returnExtendedAttributeValues, $cultureCode, $modifiedUntilDate);
        try {
            $response = $this->client->GetProductsModifiedSinceDate($request);
            return $this->response($response);
        } catch (SoapFault $fault) {
            $this->LogFaults($fault);
        }
        return;
    }
    
    /**
     * Get Sku by country
     * @param string $countryIdentifier (GB or Great Britain)
     * @param string $countryIdentifierType (RmisCode or RmisName)
     * @param DateTime $modifiedSinceDate
     * @return object or array of objects
     */
    public function getSkuPricesByCountry($countryIdentifier, $countryIdentifierType, DateTime $modifiedSinceDate=null) {
        $request = new GetSkuPricesByCountry($countryIdentifier, $countryIdentifierType, $modifiedSinceDate);
        try {
            $response = $this->client->GetSkuPricesByCountry($request);
            $this->LogLastRequest();
            return $this->response($response);
        } catch (SoapFault $fault) {
           $this->LogFaults($fault);
        }
        return;
    }
    

    /**
     * Get Sku prices by country and product
     * @param string $countryIdentifier (GB or Great Britain)
     * @param string $countryIdentifierType (RmisCode or RmisName)
     * @param string $productIdentifier  (relate to the type)
     * @param string $productIdentifierType (RmisCode, SupplierProductCode, AltCode1, AltCode2)
     * @return type
     */
    public function getSkuPricesByCountryAndProduct($countryIdentifier, $countryIdentifierType, $productIdentifier, $productIdentifierType) {
        $request = new GetSkuPricesByCountryAndProduct($countryIdentifier, $countryIdentifierType, $productIdentifier, $productIdentifierType);
        try {
            $response = $this->client->GetSkuPricesByCountryAndProduct($request);
            $this->LogLastRequest();
            return $this->response($response);
        } catch (SoapFault $fault) {
            $this->LogFaults($fault);
        }
        return;
    }
    
    /**
     * Get Sku's Price by country and identifier
     * @param string $countryIdentifier
     * @param string $countryIdentifierType (RmisCode or RmisName)
     * @param string $skuIdentifier
     * @param string $skuIdentifierType (RmisCode, PosCode, EanCode, ExistCode)
     * @return object
     */
    public function getSkuPriceByCountryAndSku($countryIdentifier, $countryIdentifierType, $skuIdentifier, $skuIdentifierType) {
        $request = new GetSkuPriceByCountryAndSku($countryIdentifier, $countryIdentifierType, $skuIdentifier, $skuIdentifierType);
        try {
            $response =  $this->client->GetSkuPriceByCountryAndSku($request);
            $this->LogLastRequest();
            return $this->response($response);
        } catch (SoapFault $fault) {
            $this->LogFaults($fault);
        }
        return;
    }
    
    
    public function getProduct($productIdentifier, $productIdentifierType, $cultureCode, $returnAnalysisValues = null, $returnExtendedAttributeValues = null) {
        $request = new GetProduct($productIdentifier, $productIdentifierType, $returnAnalysisValues, $returnExtendedAttributeValues, $cultureCode);
        try {
            $response = $this->client->GetProduct($request);
            $this->LogLastRequest();
            return $this->response($response);
        } catch (SoapFault $fault) {
            $this->LogFaults($fault);
        }
        return;
    }
    
    public function getSkuPricesBySku($skuIdentifier, $skuIdentifierType) {
        $request = new GetSkuPricesBySku($skuIdentifier, $skuIdentifierType);
        try {
            $response = $this->client->GetSkuPricesBySku($request);
            $this->LogLastRequest();
            return $this->response($response);
        } catch (SoapFault $fault) {
            $this->LogFaults($fault);
        }
        return;
    }
        
    /**
     * Get Colour
     * @param string $colourIdentifier
     * @param string $colourIdentifierType (RmisCode, RmisName, RmisShortName)
     * @param string $seasonIdentifier
     * @param string $hierarchyNodeIdentifierType (RmisCode, RmisName, RmisShortName)
     * @param string $cultureCode (en-GB etc)
     * @return object or array of objects
     */
    public function getColour($colourIdentifier, $colourIdentifierType, $seasonIdentifier, $hierarchyNodeIdentifierType, $cultureCode = null) {
        $request = new GetColour($colourIdentifier, $colourIdentifierType, $seasonIdentifier, $hierarchyNodeIdentifierType, $cultureCode);
        try {
            $response = $this->client->GetColour($request);
            $this->LogLastRequest();
            return $this->response($response);
        } catch (SoapFault $fault) {
            $this->LogFaults($fault);
        }
        return;
    }
    
    /**
     * Get Size
     * @param string $sizeIdentifier
     * @param string $sizeIdentifierType (RmisCode, RmisName, RmisShortName)
     * @param string $cultureCode (en-GB etc)
     * @return object or array of objects
     */
    public function getSize($sizeIdentifier, $sizeIdentifierType, $cultureCode = null) {
        $request = new GetSize($sizeIdentifier, $sizeIdentifierType, $cultureCode);
        try {
            $response = $this->client->GetSize($request);
            $this->LogLastRequest();
            return $this->response($response);
        } catch (SoapFault $fault) {
            $this->LogFaults($fault);
        }
        return;
    }
    
    
    public function getAllColours() {
        $request = new GetAllColours();
        try {
            $response = $this->client->GetAllColours($request);
            $this->LogLastRequest();
            return $this->response($response);
        } catch (SoapFault $fault) {
            $this->LogFaults($fault);
        }
        return;
    }
    
    public function getAllSizeGroups() {
        $request = new GetAllSizeGroups();
        try {
            $response = $this->client->GetAllSizeGroups($request);
            $this->LogLastRequest();
            return $this->response($response);
        } catch (SoapFault $fault) {
            $this->LogFaults($fault);
        }
        return;
    }
    
    public function getAllSizes() {
        $request = new GetAllSizes();
        try {
            $response = $this->client->GetAllSizes($request);
            $this->LogLastRequest();
            return $this->response($response);
        } catch (SoapFault $fault) {
            $this->LogFaults($fault);
        }
        return;
    }
    
    
    
    
    public function getAllProductAnalysisValues() {
        $request = new GetAllProductAnalysisValues();
        try {
            $response = $this->client->GetAllProductAnalysisValues($request);
            $this->LogLastRequest();
            return $this->response($response);
        } catch (SoapFault $fault) {
            $this->LogFaults($fault);
        }
        return;
    }
    
    public function getAllProductAnalysisTypes() {
        $request = new GetAllProductAnalysisTypes();
        try {
            $response = $this->client->GetAllProductAnalysisTypes($request);
            $this->LogLastRequest();
            return $this->response($response);
        } catch (SoapFault $fault) {
            $this->LogFaults($fault);
        }
        return;
    }
    
    
    
    public function getAllHierarchyNodes() {
        $request = new GetAllHierarchyNodes();
        try {
            $response = $this->client->GetAllHierarchyNodes($request);
            $this->LogLastRequest();
            return $this->response($response);
        } catch (SoapFault $fault) {
            $this->LogFaults($fault);
        }
        return;
    }
    
    public function getAllExtendedProductAttributeTypes() {
        $request = new GetAllExtendedProductAttributeTypes();
        try {
            $response = $this->client->GetAllExtendedProductAttributeTypes($request);
            $this->LogLastRequest();
            return $this->response($response);
        } catch (SoapFault $fault) {
            $this->LogFaults($fault);
        }
        return;
    }
    
    
    public function getAllSkuAnalysisTypes() {
        $request = new GetAllSkuAnalysisTypes();
        try {
            $response = $this->client->GetAllSkuAnalysisTypes($request);
            $this->LogLastRequest();
            return $this->response($response);
        } catch (SoapFault $fault) {
            $this->LogFaults($fault);
        }
        return;
    }
    
    public function getAllSkuAnalysisValues() {
        $request = new GetAllSkuAnalysisValues();
        try {
            $response = $this->client->GetAllSkuAnalysisValues($request);
            $this->LogLastRequest();
            return $this->response($response);
        } catch (SoapFault $fault) {
            $this->LogFaults($fault);
        }
        return;
    }
    
    public function getAllSuppliers() {
        $request = new GetAllSuppliers();
        try {
            $response = $this->client->GetAllSuppliers($request);
            $this->LogLastRequest();
            return $this->response($response);
        } catch (SoapFault $fault) {
            $this->LogFaults($fault);
        }
        return;
    }
    
   
    
   
    
    
}