<?php

/* 
 * Copyright Talisman Innovations Ltd. (2018). All rights reserved.
 */

namespace Talisman\EurostopBundle\Eurostop;

use Psr\Log\LoggerInterface;
use SoapFault;
use Eurostop\Fulfilment\SubmitCustomerOrder;
use Eurostop\Fulfilment\SubmittedCustomerOrder;
use Talisman\EurostopBundle\Eurostop\Base;
use Eurostop\Fulfilment\GetCustomerOrdersByModifiedDate;
use Eurostop\Fulfilment\SubmittedCustomerOrderLine;
use Eurostop\Fulfilment\ArrayOfSubmittedCustomerOrderLine;
use Eurostop\Fulfilment\CustomerAddress;
use Eurostop\Fulfilment\CustomerOrderStatus;
use Eurostop\Fulfilment\ArrayOfCustomerOrderShipment;
use Eurostop\Fulfilment\CustomerOrderShipment;
use Eurostop\Fulfilment\CustomerOrder;
use Eurostop\Fulfilment\SubmittedCustomerAddress;
use Eurostop\Fulfilment\GetCustomerOrdersByIdentifier;
use Eurostop\Fulfilment\GetCustomerOrderByIdentifier;
use DateTime;

/**
 * Eurostop Fulfilment
 * 
 */

class Fulfilment extends Base {
    
    protected $endpoint = 'fulfilment';

    public function __construct(LoggerInterface $logger, $url, $soapOptions) {
        parent::__construct($logger, $url. $this->endpoint . '?wsdl', $soapOptions);
        $this->client = $this->service();
    }
    
    ######################### FULFILMENT #########################
   
    /**
     * Submit (new or update) customer order for fulfilment
     * @param SubmittedCustomerOrder $customerOrder
     * @param string $submissionReference
     * A reference code (created by the client) for this batch of submitted data. 
     * The status of the import can later be looked up by quoting this reference. 
     * Note that the same submission reference code can be assigned to more than one batch of data.
     * @return type
     */
    public function submitCustomerOrder(SubmittedCustomerOrder $customerOrder, $submissionReference) {
        $request = new SubmitCustomerOrder($customerOrder, $submissionReference);
        try {
            $response = $this->client->SubmitCustomerOrder($request);
            $this->LogLastRequest();
            return $response;
        } catch (SoapFault $fault) {
            $this->LogFaults($fault);
        }
        return;
    }
    
    /**
     * Create a customer Order Object
     * @return SubmittedCustomerOrder
     */
    public function customerOrder () {
        return new SubmittedCustomerOrder();
    }
    
    public function customerAddress () {
        return new CustomerAddress();
    }
    
    public function submittedCustomerAddress () {
        return new SubmittedCustomerAddress();
    }
    
    
    public function customerShipment() {
        return new CustomerOrderShipment();
    }
    
    public function customerShipmentArray(array $shipments) {
        $setShipments = new ArrayOfCustomerOrderShipment();
        $setShipments->setCustomerOrderShipment($shipments);
        return $setShipments;
    }
    
    public function orderLine() {
        return new SubmittedCustomerOrderLine();
    }
    
    public function orderLineArray(array $orderLines) {
        $setOrderLines = new ArrayOfSubmittedCustomerOrderLine();
        $setOrderLines->setSubmittedCustomerOrderLine($orderLines);
        return $setOrderLines;
    }
    
        
    public static function status() {
        return new CustomerOrderStatus();
    }

    public function getCustomerOrdersByModifiedDate(DateTime $fromDate, DateTime $toDate, $orderStatus = null, $maxerOrders = null) {
        
        $request = new GetCustomerOrdersByModifiedDate($fromDate, $toDate, $orderStatus, $maxerOrders);
        try {
            $response = $this->client->GetCustomerOrdersByModifiedDate($request);
            $this->LogLastRequest();
            return $this->response($response);
        } catch (SoapFault $fault) {
            $this->LogFaults($fault);
        }
        return;
        
    }
    
    public function getCustomerOrdersByIdentifier($customerIdentifier, $customerIdentifierType) {
        $request = new GetCustomerOrdersByIdentifier($customerIdentifier, $customerIdentifierType);
        try {
            $response = $this->client->GetCustomerOrdersByIdentifier($request);
            $this->LogLastRequest();
            return $this->response($response);
        } catch (SoapFault $fault) {
            $this->LogFaults($fault);
        }
        return;
    }
    
    public function getCustomerOrderByIdentifier($customerIdentifier, $customerIdentifierType) {
        $request = new GetCustomerOrderByIdentifier($customerIdentifier, $customerIdentifierType);
        try {
            $response = $this->client->GetCustomerOrderByIdentifier($request);
            $this->LogLastRequest();
            return $this->response($response);
        } catch (SoapFault $fault) {
            $this->LogFaults($fault);
        }
        return;
    }
}