<?php
/**
 * Copyright Talisman Innovations Ltd. (2018). All rights reserved.
 */

namespace EKM;

use GuzzleHttp\HandlerStack;
use GuzzleHttp\Client;
use Psr\Log\LoggerInterface;
use League\OAuth2\Client\Token\AccessToken;
use ConnectorSupport\AccessToken\AccessTokenInterface;
use ConnectorSupport\Guzzle\Handler\OAuth2;
use ConnectorSupport\Guzzle\Handler\RateLimit;
use ConnectorSupport\Guzzle\Handler\Logger;

class ClientBuilder
{
    // retry after 1.0 seconds
    const RETRY_AFTER = 1.0;

    /**
     * @param string $clientId
     * @param string $clientSecret
     * @param AccessTokenInterface $token
     * @param null|LoggerInterface $logger
     * @return Client
     */
    public static function getClient($clientId, $clientSecret, AccessTokenInterface $token, LoggerInterface $logger = null)
    {
        $oauthStack = HandlerStack::create();

        $oauthLog = Logger::create($logger);
        $oauthStack->push($oauthLog);

        $provider = new EKMProvider([
            'clientId'              => $clientId,
            'clientSecret'          => $clientSecret,
            'redirectUri'           => '',
            'handler'               => $oauthStack
        ]);

        $clientStack = HandlerStack::create();

        $clientOauth = OAuth2::create($provider, $token);
        $clientStack->push($clientOauth);

        $clientRate = RateLimit::create(['default_retry_multiplier' => self::RETRY_AFTER]);
        $clientStack->push($clientRate);

        $clientLog = Logger::create($logger);
        $clientStack->push($clientLog);

        $client = new Client([

            'base_uri' => $provider->getBaseUrl(),
            'handler' => $clientStack

        ]);

        return $client;
    }

    /**
     * @param string $clientId
     * @param string $clientSecret
     * @param string $redirectUri
     * @return string Authorization Url
     */
    public static function getAuthorizationUrl($clientId, $clientSecret, $redirectUri)
    {
        $provider = new EKMProvider([
            'clientId'              => $clientId,
            'clientSecret'          => $clientSecret,
            'redirectUri'           => $redirectUri
        ]);

        return $provider->getAuthorizationUrl();
    }

    /**
     * @param string $clientId
     * @param string $clientSecret
     * @param string $redirectUri
     * @param string $authCode
     * @param AccessTokenInterface $token
     */
    public static function getAccessToken($clientId, $clientSecret, $redirectUri, $authCode, AccessTokenInterface $token)
    {
        $provider = new EKMProvider([
            'clientId'              => $clientId,
            'clientSecret'          => $clientSecret,
            'redirectUri'           => $redirectUri
        ]);

        /** @var AccessToken $accessToken */
        $accessToken = $provider->getAccessToken('authorization_code', [
            'code' => $authCode
        ]);

        $token->set($accessToken->jsonSerialize());
    }
}