<?php
/**
 * Copyright Talisman Innovations Ltd. (2018). All rights reserved.
 */

namespace EKM;

use GuzzleHttp\HandlerStack;
use GuzzleHttp\Client;
use League\OAuth2\Client\Provider\Exception\IdentityProviderException;
use Psr\Log\LoggerInterface;
use League\OAuth2\Client\Token\AccessToken;
use ConnectorSupport\AccessToken\AccessTokenInterface;
use ConnectorSupport\Guzzle\Handler\OAuth2;
use ConnectorSupport\Guzzle\Handler\RateLimit;
use ConnectorSupport\Guzzle\Handler\Logger;
use Symfony\Component\Lock\LockInterface;

class ClientBuilder
{
    // retry after 1.0 seconds
    const RETRY_AFTER = 1.0;

    /**
     * @param string $clientId
     * @param string $clientSecret
     * @param AccessTokenInterface $token
     * @param LockInterface $lock
     * @param null|LoggerInterface $logger
     * @return Client
     */
    public static function getClient($clientId, $clientSecret, AccessTokenInterface $token, LockInterface $lock, LoggerInterface $logger = null)
    {
        $clientRate = RateLimit::create(['default_retry_multiplier' => self::RETRY_AFTER]);
        $clientLog = Logger::create($logger);

        $oauthStack = HandlerStack::create();
        $oauthStack->push($clientRate);
        $oauthStack->push($clientLog);

        $provider = new EKMProvider([
            'clientId' => $clientId,
            'clientSecret' => $clientSecret,
            'redirectUri' => '',
            'handler' => $oauthStack
        ]);

        $clientStack = HandlerStack::create();

        $clientOauth = OAuth2::create($provider, $token, $lock, $logger);
        $clientStack->push($clientOauth);
        $clientStack->push($clientRate);
        $clientStack->push($clientLog);

        return new Client(
            [
                'base_uri' => $provider->getBaseUrl(),
                'handler' => $clientStack
            ]
        );
    }

    /**
     * @param $clientId
     * @param $clientSecret
     * @param $redirectUri
     * @param array $options
     * @return mixed
     */
    public static function getAuthorizationUrl($clientId, $clientSecret, $redirectUri, array $options = [])
    {
        $provider = new EKMProvider(
            [
                'clientId' => $clientId,
                'clientSecret' => $clientSecret,
                'redirectUri' => $redirectUri
            ]
        );

        return $provider->getAuthorizationUrl($options);
    }

    /**
     * @param string $clientId
     * @param string $clientSecret
     * @param string $redirectUri
     * @param string $authCode
     * @param AccessTokenInterface $token
     * @param LoggerInterface $logger
     * @throws IdentityProviderException
     */
    public static function getAccessToken($clientId, $clientSecret, $redirectUri, $authCode, AccessTokenInterface $token, LoggerInterface $logger = null)
    {
        $oauthStack = HandlerStack::create();

        $oauthLog = Logger::create($logger);
        $oauthStack->push($oauthLog);

        $provider = new EKMProvider([
            'clientId' => $clientId,
            'clientSecret' => $clientSecret,
            'redirectUri' => $redirectUri,
            'handler' => $oauthStack
        ]);

        /** @var AccessToken $accessToken */
        $accessToken = $provider->getAccessToken('authorization_code', [
            'code' => $authCode
        ]);

        $token->set($accessToken->jsonSerialize());
    }
}