<?php
/**
 * Defines Red61\Via\CircuitBreaker\ViaCircuitBreakerPlugin
 *
 * @author     Andrew Coulton <andrew@ingenerator.com>
 * @copyright  2014 Edinburgh International Book Festival Ltd
 * @licence    http://opensource.org/licenses/BSD-3-Clause
 */

namespace Red61\Via\CircuitBreaker;

use Red61\Via\Exception\TooManyCartsException;
use Red61\Via\Exception\ViaCircuitBreakerTrippedException;
use Red61\Via\Exception\ViaHTTPException;
use Red61\Via\Exception\ViaInternalServerErrorException;
use Red61\Via\Plugin\ViaApiCallNotification;
use Red61\Via\Plugin\ViaPluginInterface;
use Red61\Via\Plugin\ViaPluginManager;
use Red61\Via\Plugin\ViaPluginNotification;

/**
 * The ViaCircuitBreakerPlugin implements the hooks required to interface the ViaCircuitBreaker with the
 * before, success and failure API call notifications. See the package README.md for complete
 * documentation.
 *
 * @package Red61\Via\CircuitBreaker
 * @see     spec\Red61\Via\CircuitBreaker\ViaCircuitBreakerPluginSpec
 */
class ViaCircuitBreakerPlugin implements ViaPluginInterface {

	/**
	 * @var ViaCircuitBreaker
	 */
	protected $breaker;

	/**
	 * @param ViaCircuitBreaker $breaker
	 */
	public function __construct(ViaCircuitBreaker $breaker)
	{
		$this->breaker = $breaker;
	}

	/**
	 * @param ViaPluginManager $plugin_manager
	 */
	public function registerWithManager(ViaPluginManager $plugin_manager)
	{
		$plugin_manager->registerPlugin($this, NULL);
	}

	/**
	 * Receives and handles a ViaPluginNotification. The types of notifications received are
	 * controlled by the arguments you pass when registering your plugin with the ViaPluginManager.
	 *
	 * @param ViaPluginNotification $notification
	 *
	 * @return void
	 */
	public function onViaPluginNotification(ViaPluginNotification $notification)
	{
		if ( ! $notification instanceof ViaApiCallNotification)
			return;

		switch ($notification->getCallStatus())
		{
			case ViaApiCallNotification::ON_BEFORE_CALL:
				$this->throwIfCallBlocked();
				break;

			case ViaApiCallNotification::ON_CALL_FAILED:
				$this->notifyFailure($notification->getException());
				break;

			case ViaApiCallNotification::ON_CALL_SUCCESS:
				$this->breaker->reset();
				break;
		}
	}

	/**
	 * @throws ViaCircuitBreakerTrippedException
	 * @return void
	 */
	protected function throwIfCallBlocked()
	{
		if ( ! $this->breaker->requestPermissionToCall())
		{
			throw new ViaCircuitBreakerTrippedException;
		}
	}

	/**
	 * @param \Exception $exception
	 */
	protected function notifyFailure(\Exception $exception)
	{
		if ($exception instanceof TooManyCartsException)
		{
			// TooManyCarts may commonly be generated at high rate when system busy, do not treat as error
			return;
		}
		elseif (($exception instanceof ViaHTTPException) OR ($exception instanceof ViaInternalServerErrorException))
		{
			$severity = ViaCircuitBreaker::SEVERITY_CRITICAL;
		}
		else
		{
			$severity = ViaCircuitBreaker::SEVERITY_WARNING;
		}

		$this->breaker->notifyFailure($severity, $exception);
	}

}
