<?php
/**
 * Defines Red61\Via\Profiler\ViaProfilerPlugin
 *
 * @copyright  2014 Red61 Ltd
 * @licence    proprietary
 */

namespace Red61\Via\Profiler;

use Red61\Via\Plugin\ViaApiCallNotification;
use Red61\Via\Plugin\ViaPluginInterface;
use Red61\Via\Plugin\ViaPluginManager;
use Red61\Via\Plugin\ViaPluginNotification;

/**
 * The ViaProfilerPlugin provides simple profiling of the API calls made by an application.
 *
 * It is primarily intended for use during development and QA in particular on CMS or widget-based
 * sites for verification that the VIA API is being used efficiently.
 *
 * It could also form the basis for integration with realtime metrics and monitoring systems.
 *
 * [!!] Note that the execution times captured include the time spent creating the \SoapClient
 *      for the first call on a given ViaApiClient instance, and any time spent in plugins.
 *
 * @package Red61\Via\Profiler
 * @see     spec\Red61\Via\Profiler\ViaProfilerPluginSpec
 */
class ViaProfilerPlugin implements ViaPluginInterface
{
	const CALL_FAILED = 'failure';
	const CALL_CACHED = 'cached';
	const CALL_SUCCESS = 'success';

	/**
	 * @var array the raw call times, grouped by status and method
	 */
	protected $call_times = array();

	/**
	 * Handles the VIA plugin notifications and tracks the execution time of all calls.
	 *
	 * @param ViaPluginNotification $notification
	 *
	 * @return void
	 */
	public function onViaPluginNotification(ViaPluginNotification $notification)
	{
		if ( ! $notification instanceof ViaApiCallNotification) {
			return;
		}

		switch ($notification->getCallStatus()) {
			case ViaApiCallNotification::ON_CALL_SUCCESS:
				$status = $notification->shouldSkipCall() ? self::CALL_CACHED : self::CALL_SUCCESS;
				break;

			case ViaApiCallNotification::ON_CALL_FAILED:
				$status = self::CALL_FAILED;
				break;

			default:
				return;
		}

		$time = microtime(TRUE) - $notification->getStartMicrotime();

		$this->call_times[$status][$notification->getRequest()->getSoapMethodName()][] = $time;
	}

	/**
	 * Registers with the provided plugin manager - provided as a convenience to ensure the correct events
	 * are subscribed. You can equally register the plugin yourself.
	 *
	 * The profiler should ideally be registered last in your plugin chain, and definitely after any cache
	 * plugin if you wish to capture cache hits and misses separately.
	 *
	 * @param ViaPluginManager $plugin_manager
	 *
	 * @return void
	 */
	public function registerWithManager(ViaPluginManager $plugin_manager)
	{
		$plugin_manager->registerPlugin($this, array(ViaApiCallNotification::ON_CALL_SUCCESS,
													 ViaApiCallNotification::ON_CALL_FAILED));
	}

	/**
	 * Returns the tracked profiler times, grouped by status, method and statistic.
	 *
	 * For example:
	 *
	 *     array(
	 *       'success' => array(
	 *         'apiAddTicketsToBasket' => array(
	 *           'count' => 5,
	 *           'max'   => 0.130232323,
	 *           'min'   => 0.012323223,
	 *           'avg'   => 0.082323214,
	 *           'sum'   => 0.423421323,
	 *         )
	 *       'cached' => array(),
	 *       'failure' => array()
	 *     )
	 *
	 * The status keys are the values of the ViaProfilerPlugin::CALL_XXX constants.
	 *
	 * [!!] The legacy Red61_Via profiler only tracked successful calls - if migrating to this
	 *      method you will need to extract the values from the ViaProfilerPlugin::CALL_SUCCESS
	 *      key.
	 *
	 * @return array
	 */
	public function getProfilerStats()
	{
		$results = array(
			self::CALL_SUCCESS => array(),
			self::CALL_CACHED  => array(),
			self::CALL_FAILED  => array(),
		);

		foreach ($this->call_times as $status => $method_times) {
			foreach ($method_times as $method => $times) {
				$results[$status][$method] = array(
					'count' => count($times),
					'max'   => max($times),
					'min'   => min($times),
					'avg'   => array_sum($times) / count($times),
				);
			}
		}
		return $results;
	}

}
