<?php
/**
 * Notification message sent before and after an API call is made
 *
 * @author    Andrew Coulton <andrew@ingenerator.com>
 * @copyright 2014 Red61 Ltd
 * @licence   proprietary
 */

namespace Red61\Via\Plugin;

use Red61\Via\ApiRequest\ViaApiRequest;

/**
 * Instances of ViaApiCallNotification will be sent to registered plugins before and after an
 * API call is sent to VIA. To allow tracking the status of the call through the process, the
 * same instance is used for successive event notifications, and its status will be updated
 * accordingly.
 *
 * @package Red61\Via\Plugin
 */
class ViaApiCallNotification implements ViaPluginNotification
{

	const ON_BEFORE_CALL  = 'via.call.on_before_call';
	const ON_CALL_SUCCESS = 'via.call.on_call_success';
	const ON_CALL_FAILED  = 'via.call.on_call_failed';

	/**
	 * @var ViaApiRequest
	 */
	protected $request;

	/**
	 * @var mixed
	 */
	protected $response;

	/**
	 * @var string
	 */
	protected $call_status;

	/**
	 * @var \Exception
	 */
	protected $exception;

	/**
	 * @var bool
	 */
	protected $skip_call = FALSE;

	/**
	 * @var float
	 */
	protected $start_microtime;

	/**
	 * @param ViaApiRequest $request the API request about to be sent
	 */
	public function __construct(ViaApiRequest $request)
	{
		$this->request         = $request;
		$this->call_status     = self::ON_BEFORE_CALL;
		$this->start_microtime = microtime(TRUE);
	}

	/**
	 * Get the current status of the call - one of the ViaApiCallNotification::ON_ status constants.
	 * Use this value to identify whether the call has yet been sent to VIA, and if so whether it was
	 * successful or failed.
	 *
	 * @return string
	 */
	public function getCallStatus()
	{
		return $this->call_status;
	}

	/**
	 * @return ViaApiRequest the request that was (or will be) sent to VIA.
	 */
	public function getRequest()
	{
		return $this->request;
	}

	/**
	 * Call this method during an ON_BEFORE_CALL notification handler to have the ViaApiClient return the provided response
	 *
	 * This will skip sending the API call to VIA. It is intended primarily for use with caching, although could also be
	 * used to implement canned responses for requests in certain conditions.
	 *
	 * @param mixed $response
	 */
	public function skipCallAndReturn($response)
	{
		$this->response  = $response;
		$this->skip_call = TRUE;
	}

	/**
	 * Identify whether the API call should actually be made, or whether a plugin has provided a response to use instead
	 *
	 * @return bool
	 */
	public function shouldSkipCall()
	{
		return $this->skip_call;
	}

	/**
	 * The response that was returned by the client. This may also have a value during ON_BEFORE_CALL if another plugin
	 * has already called skipCallAndReturn.
	 *
	 * [!!] An ON_BEFORE_CALL plugin may force VIA to return a NULL result. To detect if the request will be (or was)
	 *      skipped you should check the shouldSkipCall method.
	 *
	 * @return mixed
	 */
	public function getResponse()
	{
		return $this->response;
	}

	/**
	 * Mark that the call was successful with the given response.
	 *
	 * [!!] This method is not intended to be called from end-user code. To inject a response (eg from a cache) you
	 *      should instead be using skipCallAndReturn().
	 *
	 * @param mixed $response
	 */
	public function _setSuccessfulResponse($response)
	{
		$this->response    = $response;
		$this->call_status = self::ON_CALL_SUCCESS;
	}

	/**
	 * Mark that a call failed with the given Exception.
	 *
	 * [!!] This method is not intended to be called from end-user code. If you need to send an exception from
	 *      your plugin you should simply throw one. Note that exceptions thrown by plugins will not automatically
	 *      be mapped to a VIAException class and will not trigger an ON_CALL_FAILED event notification.
	 *
	 * @param \Exception $exception
	 */
	public function _setFailureException(\Exception $exception)
	{
		$this->response    = NULL;
		$this->exception   = $exception;
		$this->call_status = self::ON_CALL_FAILED;
	}

	/**
	 * Return the exception that was thrown on a failed call, if any.
	 *
	 * @return \Exception
	 */
	public function getException()
	{
		return $this->exception;
	}

	/**
	 * This is the name of the notification - plugins can use this to subscribe only to particular events
	 *
	 * @return string
	 */
	public function getNotificationName()
	{
		return $this->getCallStatus();
	}

	public function getStartMicrotime()
	{
		return $this->start_microtime;
	}

} 
