<?php
/**
 * Defines Red61\Via\SoapClientFactory
 *
 * @copyright  2014 Red61 Ltd
 * @licence    proprietary
 */

namespace Red61\Via;

/**
 * Creates the instance of the \SoapClient class used to communicate with the VIA API when required.
 *
 * By delaying creation of the actual \SoapClient until just before a request is sent, methods that are satisfied by
 * the local cache or because the user has not yet created a basket can be called without the overhead of fetching,
 * caching or parsing the WSDL for the live web service improving performance and stability.
 *
 * This class also manages merging and setting the \SoapClient default options and headers that are required.
 *
 * @package Red61\Via
 * @see     spec\Red61\Via\SoapClientFactorySpec
 */
class SoapClientFactory
{

	/**
	 * @var string
	 */
	protected $client_class;

	/**
	 * @var array
	 */
	protected $soap_options;

	/**
	 * @var bool
	 */
	protected $original_xdebug_state;

	/**
	 * @var int
	 */
	protected $original_error_reporting;

	/**
	 * Create an instance of the factory. The options passed in here will be merged with the defaults before passing
	 * to the \SoapClient constructor when it is eventually created.
	 *
	 * See http://www.php.net/manual/en/soapclient.soapclient.php for the available options.
	 *
	 * @param array  $soap_options custom options for the \SoapClient.
	 * @param string $client_class the class to create - it must be API-compatible with \SoapClient
	 */
	public function __construct($soap_options, $client_class = '\SoapClient')
	{
		$this->soap_options = $this->mergeDefaultOptions($soap_options);
		$this->client_class = $client_class;
	}

	/**
	 * @param array $soap_options
	 * @return array
	 */
	protected function mergeDefaultOptions($soap_options)
	{
		$defaults = array(
			'classmap'    => ViaClassmap::$dataClassmap,
			'compression' => SOAP_COMPRESSION_ACCEPT | SOAP_COMPRESSION_GZIP,
			'exceptions'  => TRUE,
			'features'    => SOAP_SINGLE_ELEMENT_ARRAYS,
			'trace'       => FALSE
		);

		if (isset($soap_options['classmap'])) {
			$soap_options['classmap'] = array_merge($defaults['classmap'], $soap_options['classmap']);
		}

		return array_merge($defaults, $soap_options);
	}

	/**
	 * @param string $wsdl
	 *
	 * @return \SoapClient
	 */
	public function make($wsdl)
	{
		$client = $this->makeClientOrThrow($wsdl);

		$remote_addr = isset($_SERVER['REMOTE_ADDR']) ? $_SERVER['REMOTE_ADDR'] : '127.0.0.1';
		$client->__setSoapHeaders(new \SoapHeader('com.red61.via.api', 'REMOTE_ADDR', $remote_addr));

		return $client;
	}

	/**
	 * @param string $wsdl
	 * @throws \Exception on failure to create client
	 *
	 * @return \SoapClient
	 */
	protected function makeClientOrThrow($wsdl)
	{
		$this->disableXdebugAndErrorReporting();
		try {
			$class  = $this->client_class;
			$client = new $class($wsdl, $this->soap_options);
			$this->restoreXdebugAndErrorReporting();
			return $client;
		} catch (\Exception $e) {
			$this->restoreXdebugAndErrorReporting();
			throw $e;
		}
	}

	/**
	 * Resolves known SOAP issues that can cause fatal errors if a server is unreachable, preventing them being handled
	 *
	 * There is a known conflict with xdebug and SOAP (http://bugs.xdebug.org/bug_view_page.php?bug_id=00000609) which
	 * causes a PHP fatal error internally because of the way both extensions attempt to modify PHP's error handling.
	 *
	 * SOAP also triggers a fatal error - even if set to use exceptions - if the WSDL is unreachable. In applications
	 * that are using a custom error handler (for eg to rethrow an ErrorException) this error will be caught and
	 * dispatched globally preventing the calling code from intercepting the SoapFault that would otherwise bubble up.
	 *
	 * Therefore we modify the PHP environment immediately before and after attempting to create the SoapClient.
	 *
	 * @return void
	 * @see    restoreXdebugAndErrorReporting
	 */
	protected function disableXdebugAndErrorReporting()
	{
		if (function_exists('xdebug_disable') AND function_exists('xdebug_is_enabled')) {
			$this->original_xdebug_state = xdebug_is_enabled();
			xdebug_disable();
		} else {
			$this->original_xdebug_state = NULL;
		}
		$this->original_error_reporting = error_reporting(0);
	}

	/**
	 * Reinstate the environment modified by disableXdebugAndErrorReporting - see that method for explanation
	 *
	 * @return void
	 * @see disableXdebugAndErrorReporting
	 */
	protected function restoreXdebugAndErrorReporting()
	{
		if ($this->original_xdebug_state) {
			xdebug_enable();
		}
		error_reporting($this->original_error_reporting);
	}

}
