<?php
/**
 * Defines ViaCoreCircuitBreakerSpec - specifications for Red61\Via\CircuitBreaker\ViaCoreCircuitBreaker
 *
 * @author    Andrew Coulton <andrew@ingenerator.com>
 * @copyright 2014 Edinburgh International Book Festival Ltd
 * @licence   http://opensource.org/licenses/BSD-3-Clause
 */

namespace spec\Red61\Via\CircuitBreaker;

use Prophecy\Argument;
use Red61\Via\CircuitBreaker\ViaCircuitBreaker;
use Red61\Via\CircuitBreaker\ViaCircuitBreakerListener;
use Red61\Via\CircuitBreaker\ViaCircuitBreakerStorage;
use Red61\Via\Exception\ViaException;
use spec\ObjectBehavior;

/**
 *
 * @see Red61\Via\CircuitBreaker\ViaCoreCircuitBreaker
 */
class ViaCoreCircuitBreakerSpec extends ObjectBehavior {

	const RETRY_INTERVAL = 90;

	/**
	 * Use $this->subject to get proper type hinting for the subject class
	 *
	 * @var \Red61\Via\CircuitBreaker\ViaCoreCircuitBreaker
	 */
	protected $subject;

	/**
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreakerStorage $storage
	 */
	function let($storage)
	{
		$this->beConstructedWith($storage, array());
		$storage->wasFirstToTrip()->willReturn(FALSE);
	}

	function it_is_an_initializable_circuit_breaker_implementation()
	{
		$this->subject->shouldHaveType('Red61\Via\CircuitBreaker\ViaCoreCircuitBreaker');
		$this->subject->shouldBeAnInstanceOf('Red61\Via\CircuitBreaker\ViaCircuitBreaker');
	}

	/**
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreakerStorage $storage
	 */
	function it_is_tripped_if_storage_is_tripped($storage)
	{
		$storage->isTripped()->willReturn(FALSE);
		$this->subject->isTripped()->shouldBe(FALSE);

		$storage->isTripped()->willReturn(TRUE);
		$this->subject->isTripped()->shouldBe(TRUE);
	}

	/**
	 * Applications - especially legacy sites - may exhibit unexpected behaviour if the breaker
	 * flickers during the execution of a single request. Therefore, once the breaker trips for
	 * a single instance it should stay tripped unless explicitly reset. Similarly only the first
	 * call during a request can be sent, even if the retry interval clears during the period.
	 *
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreakerStorage $storage
	 */
	function it_stays_tripped_once_checked_even_if_storage_resets($storage)
	{
		$storage->isTripped()->willReturn(TRUE);
		$this->subject->isTripped()->shouldBe(TRUE);

		$storage->isTripped()->willReturn(FALSE);
		$this->subject->isTripped()->shouldBe(TRUE);
	}

	/**
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreakerStorage $storage
	 */
	function it_is_no_longer_tripped_and_storage_is_reset_after_reset($storage)
	{
		$this->givenTrippedAndResettable($storage);
		$this->subject->isTripped()->shouldBe(TRUE);
		$this->subject->reset();
		$this->subject->isTripped()->shouldBe(FALSE);
	}


	/**
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreakerStorage $storage
	 */
	function it_grants_permission_to_call_if_not_tripped($storage)
	{
		$storage->isTripped()->willReturn(FALSE);
		$this->subject->requestPermissionToCall()->shouldBe(TRUE);
	}

	/**
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreakerStorage $storage
	 */
	function it_refuses_permission_to_call_if_tripped_and_retried_within_interval($storage)
	{
		$this->givenTrippedAndRecentlyRetried($storage);
		$this->subject->requestPermissionToCall()->shouldBe(FALSE);
	}

	/**
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreakerStorage $storage
	 */
	function its_retry_interval_is_configurable($storage)
	{
		$this->beConstructedWith($storage, array('retry_interval' => 90));
		$this->givenTrippedAndRecentlyRetried($storage);
		$this->subject->requestPermissionToCall();
		$storage->wasRetriedWithinSeconds(90)->shouldHaveBeenCalled();
	}

	/**
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreakerStorage $storage
	 */
	function it_grants_permission_to_call_after_each_retry_interval_if_lock_available($storage)
	{
		$this->givenTrippedAndNotRetried($storage);
		$this->givenRetryLockAvailability($storage, TRUE);
		$this->subject->requestPermissionToCall()->shouldBe(TRUE);
	}

	/**
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreakerStorage $storage
	 */
	function it_refuses_retry_permission_if_lock_not_available($storage)
	{
		$this->givenTrippedAndNotRetried($storage);
		$this->givenRetryLockAvailability($storage, FALSE);
		$this->subject->requestPermissionToCall()->shouldBe(FALSE);
	}

	/**
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreakerStorage $storage
	 */
	function its_retry_lock_lifetime_is_configurable($storage)
	{
		$this->beConstructedWith($storage, array('retry_lock_lifetime' => 10));
		$this->givenTrippedAndNotRetried($storage);
		$this->givenRetryLockAvailability($storage, FALSE);
		$this->subject->requestPermissionToCall();
		$storage->tryLockForRetry(10)->shouldHaveBeenCalled();
	}

	/**
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreakerStorage $storage
	 */
	function it_refuses_permission_for_all_calls_after_refusal_even_if_storage_resets($storage)
	{
		$this->givenTrippedAndRecentlyRetried($storage);
		$this->subject->requestPermissionToCall()->shouldBe(FALSE);

		$this->givenStorageReset($storage);
		$this->subject->requestPermissionToCall()->shouldBe(FALSE);
	}

	/**
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreakerStorage $storage
	 */
	function it_allows_calls_after_refusal_if_this_instance_is_reset($storage)
	{
		$this->givenTrippedAndResettable($storage);
		$this->givenTrippedAndRecentlyRetried($storage);

		$this->subject->requestPermissionToCall()->shouldBe(FALSE);
		$this->subject->reset();
		$this->subject->requestPermissionToCall()->shouldBe(TRUE);
	}

	/**
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreakerStorage $storage
	 */
	function it_increments_error_rate_by_severity($storage)
	{
		$severity = uniqid();
		$storage->incrementErrorRate($severity)->shouldBeCalled();
		$this->subject->notifyFailure($severity, new DummyViaException);
	}

	/**
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreakerStorage $storage
	 */
	function it_does_not_trip_if_error_severity_rate_not_configured($storage)
	{
		$this->beConstructedWith($storage, array());
		$storage->incrementErrorRate(ViaCircuitBreaker::SEVERITY_CRITICAL)->shouldBeCalled();
		$this->subject->notifyFailure(ViaCircuitBreaker::SEVERITY_CRITICAL, new DummyViaException);
		$storage->trip()->shouldNotHaveBeenCalled();
	}

	/**
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreakerStorage $storage
	 */
	function it_trips_immediately_if_severity_rate_limit_set_to_zero($storage)
	{
		$severity = uniqid();
		$this->givenConfiguredRateLimit($storage, $severity, 0);
		$this->givenCurrentRate($storage, $severity, 1);
		$storage->trip()->shouldBeCalled();
		$this->subject->notifyFailure($severity, new DummyViaException);
	}

	/**
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreakerStorage $storage
	 */
	function it_does_not_trip_if_error_severity_rate_below_limit($storage)
	{
		$severity = uniqid();
		$this->givenConfiguredRateLimit($storage, $severity, 2);
		$this->givenCurrentRate($storage, $severity, 1);
		$this->subject->notifyFailure($severity, new DummyViaException);
		$storage->trip()->shouldNotHaveBeenCalled();
	}

	/**
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreakerStorage $storage
	 */
	function it_trips_if_error_severity_rate_over_limit($storage)
	{
		$severity = uniqid();
		$this->givenConfiguredRateLimit($storage, $severity, 2);
		$this->givenCurrentRate($storage, $severity, 3);
		$storage->trip()->shouldBeCalled();
		$this->subject->notifyFailure($severity, new DummyViaException);
	}

	/**
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreakerStorage $storage
	 */
	function it_marks_as_retried_on_any_configured_error_while_tripped($storage)
	{
		$this->givenConfiguredRateLimit($storage, ViaCircuitBreaker::SEVERITY_WARNING, 90);
		$storage->isTripped()->willReturn(TRUE);
		$storage->markRetried()->shouldBeCalled();
		$this->subject->notifyFailure(ViaCircuitBreaker::SEVERITY_WARNING, new DummyViaException);
		$storage->trip()->shouldNotHaveBeenCalled();
	}

	/**
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreakerStorage $storage
	 */
	function it_does_not_mark_retried_while_tripped_if_error_severity_rate_not_configured($storage)
	{
		$this->beConstructedWith($storage, array());
		$storage->isTripped()->willReturn(TRUE);
		$storage->incrementErrorRate(ViaCircuitBreaker::SEVERITY_CRITICAL)->willReturn(NULL);
		$this->subject->notifyFailure(ViaCircuitBreaker::SEVERITY_CRITICAL, new DummyViaException);
		$storage->markRetried()->shouldNotHaveBeenCalled();
		$storage->trip()->shouldNotHaveBeenCalled();
	}

	/**
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreakerStorage  $storage
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreakerListener $listener
	 */
	function it_notifies_if_first_to_trip($storage, $listener)
	{
		$this->givenConfiguredRateLimit($storage, ViaCircuitBreaker::SEVERITY_CRITICAL, 0);
		$this->givenCurrentRate($storage, ViaCircuitBreaker::SEVERITY_CRITICAL, 1);
		$this->givenWillBeFirstToTrip($storage, TRUE);

		$this->subject->setListener($listener);
		$exception = new DummyViaException;
		$this->subject->notifyFailure(ViaCircuitBreaker::SEVERITY_CRITICAL, $exception);
		$listener->onTripped(ViaCircuitBreaker::SEVERITY_CRITICAL, $exception, 1)->shouldHaveBeenCalled();
	}

	/**
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreakerStorage  $storage
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreakerListener $listener
	 */
	function it_does_not_notify_if_not_first_to_trip($storage, $listener)
	{
		$this->givenConfiguredRateLimit($storage, ViaCircuitBreaker::SEVERITY_CRITICAL, 0);
		$this->givenCurrentRate($storage, ViaCircuitBreaker::SEVERITY_CRITICAL, 1);
		$this->givenWillBeFirstToTrip($storage, TRUE);

		$this->subject->setListener($listener);
		$exception = new DummyViaException;
		$this->subject->notifyFailure(ViaCircuitBreaker::SEVERITY_CRITICAL, $exception);
		$listener->onTripped(ViaCircuitBreaker::SEVERITY_CRITICAL, $exception, 1)->shouldHaveBeenCalled();
	}

	/**
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreakerStorage  $storage
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreakerListener $listener
	 */
	function it_notifies_on_reset($storage, $listener)
	{
		$storage->isTripped()->willReturn(TRUE);
		$storage->reset()->shouldBeCalled();
		$this->subject->setListener($listener);
		$this->subject->reset();
		$listener->onReset()->shouldHaveBeenCalled();
	}

	/**
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreakerStorage  $storage
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreakerListener $listener
	 */
	function it_does_not_notify_on_reset_if_not_tripped($storage, $listener)
	{
		$storage->isTripped()->willReturn(FALSE);
		$this->subject->setListener($listener);
		$this->subject->reset();
		$listener->onReset(Argument::any())->shouldNotHaveBeenCalled();
	}

	/**
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreakerStorage  $storage
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreakerListener $listener
	 */
	function it_notifies_when_call_is_blocked($storage, $listener)
	{
		$this->givenTrippedAndRecentlyRetried($storage);
		$this->subject->setListener($listener);
		$this->subject->requestPermissionToCall()->shouldBe(FALSE);
		$listener->onCallBlocked()->shouldHaveBeenCalled();
	}

	/**
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreakerStorage  $storage
	 * @param \Red61\Via\CircuitBreaker\ViaCircuitBreakerListener $listener
	 */
	function it_does_not_notify_when_call_is_allowed($storage, $listener)
	{
		$storage->isTripped()->willReturn(FALSE);
		$this->subject->setListener($listener);
		$this->subject->requestPermissionToCall()->shouldBe(TRUE);
		$listener->onCallBlocked()->shouldNotHaveBeenCalled();
	}

	/**
	 * @param ViaCircuitBreakerStorage $storage
	 * @param string                   $severity
	 * @param int                      $limit
	 */
	protected function givenConfiguredRateLimit($storage, $severity, $limit)
	{
		$this->beConstructedWith(
			$storage,
			array(
				'severity_rate_limits' => array(
					$severity => $limit
				)
			)
		);
		$storage->incrementErrorRate($severity)->willReturn(NULL);
	}

	/**
	 * @param ViaCircuitBreakerStorage $storage
	 * @param string                   $severity
	 * @param int                      $rate
	 */
	protected function givenCurrentRate($storage, $severity, $rate)
	{
		$storage->isTripped()->willReturn(FALSE);
		$storage->getErrorsPerMinute($severity)->willReturn($rate);
	}

	/**
	 * @param ViaCircuitBreakerStorage $storage
	 */
	protected function givenTrippedAndRecentlyRetried($storage)
	{
		$storage->isTripped()->willReturn(TRUE);
		$storage->wasRetriedWithinSeconds(Argument::type('integer'))->willReturn(TRUE);
	}

	/**
	 * @param ViaCircuitBreakerStorage $storage
	 */
	protected function givenTrippedAndNotRetried($storage)
	{
		$storage->isTripped()->willReturn(TRUE);
		$storage->wasRetriedWithinSeconds(Argument::type('integer'))->willReturn(FALSE);
	}

	/**
	 * @param ViaCircuitBreakerStorage $storage
	 */
	protected function givenTrippedAndResettable($storage)
	{
		$storage->isTripped()->willReturn(TRUE);
		$storage->reset()->will(
			function ($args, $storage)
			{
				$storage->isTripped()->willReturn(FALSE);
			}
		);
	}

	/**
	 * @param ViaCircuitBreakerStorage $storage
	 */
	protected function givenStorageReset($storage)
	{
		$storage->isTripped()->willReturn(FALSE);
		$storage->wasRetriedWithinSeconds(Argument::type('integer'))->willReturn(FALSE);
	}

	/**
	 * @param ViaCircuitBreakerStorage $storage
	 * @param bool                     $lockable
	 */
	protected function givenRetryLockAvailability($storage, $lockable)
	{
		$storage->tryLockForRetry(Argument::type('integer'))->willReturn($lockable);
	}

	/**
	 * @param ViaCircuitBreakerStorage $storage
	 * @param bool                     $first
	 */
	protected function givenWillBeFirstToTrip($storage, $first)
	{
		$storage->trip()->shouldBeCalled();
		$storage->wasFirstToTrip()->willReturn($first);
	}

}


class DummyViaException extends \Exception implements ViaException {}
