<?php
/**
 * Defines Red61\Via\Cache\ViaCachePlugin
 *
 * @copyright  2014 Red61 Ltd
 * @licence    proprietary
 */


namespace Red61\Via\Cache;

use Red61\Via\ApiRequest\CacheTTLOverridingRequest;
use Red61\Via\ApiRequest\ViaApiRequest;
use Red61\Via\Plugin\ViaApiCallNotification;
use Red61\Via\Plugin\ViaPluginInterface;
use Red61\Via\Plugin\ViaPluginNotification;

/**
 * Manages caching of VIA responses and handles the notifications from ViaApiClient to intercept
 * API calls as required.
 *
 * The actual caching is handled by drivers, with this class managing TTL and whether or not the
 * cache is active.
 *
 * @package Red61\Via\Cache
 * @see     spec\Red61\Via\Cache\ViaCachePluginSpec
 */
class ViaCachePlugin implements ViaPluginInterface
{

	/**
	 * @var ViaCacheDriver
	 */
	protected $driver;

	/**
	 * @var ViaCacheKeyGenerator
	 */
	protected $key_generator;

	/**
	 * @var int[]
	 */
	protected $method_ttls = array();

	/**
	 * @var bool if true, will lookup from cache regardless of the user's http headers
	 */
	protected $ignore_http_headers = FALSE;

	/**
	 * Create an instance which will then be registered with the ViaPluginManager.
	 *
	 * By default cache lookups will be skipped if the no-cache HTTP header is present (but successful
	 * responses will still be stored). This allows box office staff to easily refresh any invalid cached
	 * data by simply force-refreshing the page in their browser.
	 *
	 * For high traffic sites, you may wish to disable this option as it can lead to heavy load during any
	 * period of slow or error responses if end users attempt to refresh their page rapidly. If so, pass
	 * `ignore_http_headers` => TRUE in the $options argument. You should then implement an alternative means
	 * of staff forcing a cache refresh as required.
	 *
	 * Alternatively, extend this class and override the isLookupDisabled method.
	 *
	 * @param ViaCacheDriver       $driver
	 * @param ViaCacheKeyGenerator $key_generator
	 * @param array                $options
	 */
	public function __construct(ViaCacheDriver $driver, ViaCacheKeyGenerator $key_generator, $options = array())
	{
		$this->driver        = $driver;
		$this->key_generator = $key_generator;
		$this->method_ttls   = ViaCacheTTLConfig::$default_ttl;

		$this->ignore_http_headers = isset($options['ignore_http_headers']) ? $options['ignore_http_headers'] : FALSE;
	}

	/**
	 * Called before and after a VIA api call to manage cache lookups and storage of responses
	 *
	 * @param ViaPluginNotification $notification
	 *
	 * @return void
	 */
	public function onViaPluginNotification(ViaPluginNotification $notification)
	{
		if (!$notification instanceof ViaApiCallNotification) {
			return;
		}

		switch ($notification->getCallStatus()) {
			case ViaApiCallNotification::ON_BEFORE_CALL:
				$this->onBeforeCall($notification);
				break;

			case ViaApiCallNotification::ON_CALL_SUCCESS:
				$this->onSuccessfulCall($notification);
				break;
		}
	}

	/**
	 * If there is a TTL defined for the method and cache lookups are not disabled, attempt to fetch from cache
	 *
	 * @param ViaApiCallNotification $notification
	 *
	 * @return void
	 */
	protected function onBeforeCall(ViaApiCallNotification $notification)
	{
		if ($this->isLookupDisabled()) {
			return;
		}

		$request = $notification->getRequest();
		if ( ! $ttl = $this->getRequestTTL($request)) {
			return;
		}

		$key = $this->key_generator->get_key($request);
		if ($response = $this->driver->lookup($request->getSoapMethodName(), $key, $ttl)) {
			$notification->skipCallAndReturn($response);
		}
	}

	/**
	 * Check whether cache lookup has been disabled - by default, will not lookup if HTTP cache headers are present
	 *
	 * @return bool
	 */
	protected function isLookupDisabled()
	{
		if (
			( ! $this->ignore_http_headers)
			AND isset($_SERVER['HTTP_CACHE_CONTROL'])
			AND ($_SERVER['HTTP_CACHE_CONTROL'] === 'no-cache')
		) {
			return TRUE;
		} else {
			return FALSE;
		}
	}

	/**
	 * Get the TTL applicable to this request]
	 *
	 * @param ViaApiRequest $request
	 *
	 * @return int
	 */
	protected function getRequestTTL(ViaApiRequest $request)
	{
		if ($request instanceof CacheTTLOverridingRequest) {
			return $request->getCacheTtl();
		}

		$name = $request->getSoapMethodName();
		if (isset($this->method_ttls[$name])) {
			return $this->method_ttls[$name];
		} else {
			return 0;
		}
	}

	/**
	 * If the request was a live API call and there is a TTL defined for the method, store it in the cache
	 *
	 * @param ViaApiCallNotification $notification
	 *
	 * @return void
	 */
	protected function onSuccessfulCall(ViaApiCallNotification $notification)
	{
		if ($notification->shouldSkipCall()) {
			return;
		}

		$request = $notification->getRequest();
		if ( ! $ttl = $this->getRequestTTL($request)) {
			return;
		}

		$key = $this->key_generator->get_key($request);
		$this->driver->store($request::SOAP_METHOD_NAME, $key, $notification->getResponse(), $ttl);
	}

	/**
	 * Override the default cache ttl for individual API requests
	 *
	 * [!!] Changes to the default TTL should be made with great care and only after consultation with Red61.
	 *      You should NEVER reduce the default TTL on any method without discussing your use case with Red61.
	 *      VIA also internally caches API responses, and altering the default TTLs in your application may
	 *      cause unexpected user behaviour.
	 *
	 * Call this method like:
	 *
	 *    $cache->overrideMethodTtls(array(\Red61\Via\ApiRequest\apiGetEventsRequest::SOAP_METHOD_NAME => 86400));
	 *
	 * Note that the actual values of the method names are not guaranteed over time and you should use the constants
	 * defined on the relevant request object wherever possible.
	 *
	 * @param int[] $method_ttls an array of SOAP method name and the TTL in seconds for the methods to override
	 *
	 * @see ViaCacheTTLConfig::$method_ttls
	 * @return void
	 */
	public function overrideMethodTtls($method_ttls)
	{
		$this->method_ttls = array_merge($this->method_ttls, $method_ttls);
	}

}
