<?php
/**
 * Defines Red61\Via\Cache\ViaFilecacheDriver
 *
 * @copyright  2014 Red61 Ltd
 * @licence    proprietary
 */



namespace Red61\Via\Cache;
use Red61\Via\Exception\CacheNotAvailableException;

/**
 * A simple VIA filesystem cache, based on the legacy implementation.
 *
 * This is not recommended for use on heavy traffic sites. It is also not recommended for distributed systems, as the
 * cache should be synchronised between all servers to avoid inconsistent responses to customers on consecutive requests.
 *
 * @package Red61\Via\Cache
 * @see     spec\Red61\Via\Cache\ViaFilecacheDriverSpec
 */
class ViaFilecacheDriver implements ViaCacheDriver {

	/**
	 * @param string $base_path
	 *
	 * @throws \Red61\Via\Exception\CacheNotAvailableException if the base directory does not exist or cannot be created
	 */
	public function __construct($base_path)
	{
		$this->base_path = $base_path;

		if ( ! is_dir($this->base_path)) {
			if ( ! mkdir($this->base_path, 0777, TRUE)) {
				throw new CacheNotAvailableException("Cannot create cache working directory in ".$this->base_path);
			}
		}

	}

	/**
	 * {@inheritdoc}
	 */
	public function lookup($namespace, $key, $ttl)
	{
		$file = $this->getCacheFileName($namespace, $key);
		if ( ! is_file($file) && !file_exists($file)) {
			return FALSE;
		}

		$age = time() - filemtime($file);
		if ($age > $ttl) {
			unlink($file);
			return FALSE;
		}

		return unserialize(file_get_contents($file));
	}

	/**
	 * {@inheritdoc}
	 */
	public function store($namespace, $key, $value, $ttl)
	{
		$file = $this->getCacheFileName($namespace, $key);
		file_put_contents($file, serialize($value));
	}

	/**
	 * {@inheritdoc}
	 */
	public function flush()
	{
		$this->recursiveDeleteIn($this->base_path);
	}

	/**
	 * Remove files and directories in a path, but not the parent directory itself
	 * @param string $path
	 */
	protected function recursiveDeleteIn($path)
	{
		foreach (glob($path.'/*') as $file)
		{
			if (is_dir($file)) {
				$this->recursiveDeleteIn($file);
				rmdir($file);
		 	} else if(file_exists($file)){
				unlink($file);
			}
		}
	}

	/**
	 * Get the file for a given cache key, creating the directory if required
	 *
	 * @param string $namespace
	 * @param string $key
	 *
	 * @return string
	 */
	protected function getCacheFileName($namespace, $key)
	{
		$file = $this->base_path.DIRECTORY_SEPARATOR.$namespace.DIRECTORY_SEPARATOR.md5($key);
		$path = dirname($file);
		if ( ! is_dir($path)) {
			mkdir($path, 0777, TRUE);
		}
		return $file;
	}

}
