<?php
/**
 * Defines ViaProfilerPluginSpec - specifications for Red61\Via\Profiler\ViaProfilerPlugin
 *
 * @copyright  2014 Red61 Ltd
 * @licence    proprietary
 */

namespace spec\Red61\Via\Profiler;

use Red61\Via\ApiRequest\apiAddTicketsToBasketRequest;
use Red61\Via\ApiRequest\apiGetEventsRequest;
use Red61\Via\ApiRequest\apiSetAffiliateRequest;
use Red61\Via\ApiRequest\ViaApiRequest;
use Red61\Via\Plugin\ViaApiCallNotification;
use Red61\Via\Profiler\ViaProfilerPlugin;
use Red61\Via\RawGenerated\ApiRequest\apiAddSchemeToBasketRequest;
use spec\ObjectBehavior;
use Prophecy\Argument;

/**
 *
 * @see Red61\Via\Profiler\ViaProfilerPlugin
 */
class ViaProfilerPluginSpec extends ObjectBehavior
{
    /**
     * Use $this->subject to get proper type hinting for the subject class
     * @var \Red61\Via\Profiler\ViaProfilerPlugin
     */
	protected $subject;

	function it_is_initializable()
    {
		$this->subject->shouldHaveType('Red61\Via\Profiler\ViaProfilerPlugin');
	}

	function it_is_a_via_plugin()
	{
		$this->subject->shouldHaveType('Red61\Via\Plugin\ViaPluginInterface');
	}

	/**
	 * @param \Red61\Via\Plugin\ViaPluginManager $plugin_manager
	 */
	function it_can_register_with_a_plugin_manager($plugin_manager)
	{
		$this->subject->registerWithManager($plugin_manager);

		$plugin_manager->registerPlugin(
			$this->subject,
			array(ViaApiCallNotification::ON_CALL_FAILED, ViaApiCallNotification::ON_CALL_SUCCESS)
		);
	}

	function it_gives_stats_broken_down_by_success_cached_and_failure()
	{
		$this->subject->getProfilerStats()->shouldBe(array(
			'success' => array(),
			'cached'  => array(),
			'failure' => array()
		));
	}

	function it_tracks_the_number_of_calls_by_status_and_method()
	{
		$this->given_notified_success_after_ms(new apiGetEventsRequest, 10);
		$this->given_notified_success_after_ms(new apiSetAffiliateRequest, 10);
		$this->given_notified_cached_after_ms(new apiGetEventsRequest, 10);
		$this->given_notified_cached_after_ms(new apiGetEventsRequest, 10);
		$this->given_notified_failed_after_ms(new apiAddTicketsToBasketRequest, 10);

		$stats = $this->subject->getProfilerStats();

		$stats[ViaProfilerPlugin::CALL_SUCCESS][apiGetEventsRequest::SOAP_METHOD_NAME]['count']->shouldBe(1);
		$stats[ViaProfilerPlugin::CALL_SUCCESS][apiSetAffiliateRequest::SOAP_METHOD_NAME]['count']->shouldBe(1);
		$stats[ViaProfilerPlugin::CALL_CACHED][apiGetEventsRequest::SOAP_METHOD_NAME]['count']->shouldBe(2);
		$stats[ViaProfilerPlugin::CALL_FAILED][apiAddTicketsToBasketRequest::SOAP_METHOD_NAME]['count']->shouldBe(1);
	}

	function it_tracks_the_maximum_time_by_status_and_method()
	{
		$this->given_notified_success_after_ms(new apiGetEventsRequest, 90);
		$this->given_notified_success_after_ms(new apiSetAffiliateRequest, 40);
		$this->given_notified_cached_after_ms(new apiGetEventsRequest, 30);
		$this->given_notified_cached_after_ms(new apiGetEventsRequest, 10);
		$this->given_notified_failed_after_ms(new apiAddTicketsToBasketRequest, 10);

		$this->subject->getProfilerStats()->shouldHaveRoughStats(
			'max',
			array(
				ViaProfilerPlugin::CALL_SUCCESS => array(
					apiGetEventsRequest::SOAP_METHOD_NAME => 0.09,
					apiSetAffiliateRequest::SOAP_METHOD_NAME => 0.04,
				),
				ViaProfilerPlugin::CALL_CACHED => array(
					apiGetEventsRequest::SOAP_METHOD_NAME => 0.03
				),
				ViaProfilerPlugin::CALL_FAILED => array(
					apiAddTicketsToBasketRequest::SOAP_METHOD_NAME => 0.01
				)
			)
		);
	}

	function it_tracks_the_minimum_time_by_status_and_method()
	{
		$this->given_notified_success_after_ms(new apiGetEventsRequest, 90);
		$this->given_notified_success_after_ms(new apiSetAffiliateRequest, 40);
		$this->given_notified_cached_after_ms(new apiGetEventsRequest, 30);
		$this->given_notified_cached_after_ms(new apiGetEventsRequest, 10);
		$this->given_notified_failed_after_ms(new apiAddTicketsToBasketRequest, 10);

		$this->subject->getProfilerStats()->shouldHaveRoughStats(
			'min',
			array(
				ViaProfilerPlugin::CALL_SUCCESS => array(
					apiGetEventsRequest::SOAP_METHOD_NAME => 0.09,
					apiSetAffiliateRequest::SOAP_METHOD_NAME => 0.04,
				),
				ViaProfilerPlugin::CALL_CACHED => array(
					apiGetEventsRequest::SOAP_METHOD_NAME => 0.01
				),
				ViaProfilerPlugin::CALL_FAILED => array(
					apiAddTicketsToBasketRequest::SOAP_METHOD_NAME => 0.01
				)
			)
		);
	}

	function it_tracks_the_average_time_by_status_and_method()
	{
		$this->given_notified_success_after_ms(new apiGetEventsRequest, 100);
		$this->given_notified_success_after_ms(new apiGetEventsRequest, 20);
		$this->given_notified_success_after_ms(new apiSetAffiliateRequest, 60);
		$this->given_notified_success_after_ms(new apiSetAffiliateRequest, 40);
		$this->given_notified_cached_after_ms(new apiGetEventsRequest, 30);
		$this->given_notified_cached_after_ms(new apiGetEventsRequest, 10);
		$this->given_notified_failed_after_ms(new apiAddTicketsToBasketRequest, 10);

		$this->subject->getProfilerStats()->shouldHaveRoughStats(
			'avg',
			array(
				ViaProfilerPlugin::CALL_SUCCESS => array(
					apiGetEventsRequest::SOAP_METHOD_NAME    => 0.06,
					apiSetAffiliateRequest::SOAP_METHOD_NAME => 0.05,
				),
				ViaProfilerPlugin::CALL_CACHED  => array(
					apiGetEventsRequest::SOAP_METHOD_NAME => 0.02
				),
				ViaProfilerPlugin::CALL_FAILED  => array(
					apiAddTicketsToBasketRequest::SOAP_METHOD_NAME => 0.01
				)
			)
		);
	}



	/**
	 * @param ViaApiRequest $request
	 * @param int        $time_ms
	 */
	protected function given_notified_success_after_ms(ViaApiRequest $request, $time_ms)
	{
		$notification = new ViaApiCallNotificationWithFakableTime($request);
		$notification->_setSuccessfulResponse(TRUE);
		$notification->fakeStartMicrotimeAgo($time_ms);
		$this->subject->onViaPluginNotification($notification);
	}

	/**
	 * @param ViaApiRequest $request
	 * @param int        $time_ms
	 */
	protected function given_notified_cached_after_ms(ViaApiRequest $request, $time_ms)
	{
		$notification = new ViaApiCallNotificationWithFakableTime($request);
		$notification->skipCallAndReturn(TRUE);
		$notification->_setSuccessfulResponse(TRUE);
		$notification->fakeStartMicrotimeAgo($time_ms);
		$this->subject->onViaPluginNotification($notification);
	}

	/**
	 * @param ViaApiRequest $request
	 * @param int        $time_ms
	 */
	protected function given_notified_failed_after_ms(ViaApiRequest $request, $time_ms)
	{
		$notification = new ViaApiCallNotificationWithFakableTime($request);
		$notification->_setFailureException(new \Exception('Something'));
		$notification->fakeStartMicrotimeAgo($time_ms);
		$this->subject->onViaPluginNotification($notification);
	}

	public function getMatchers()
	{
		$matchers = parent::getMatchers();
		$matchers['haveRoughStats'] = function ($actual, $statistic, $expected) {
			foreach ($expected as $status => $methods) {
				foreach ($methods as $method_name => $expected_value) {
					$actual_value    = $actual[$status][$method_name][$statistic];
					$time_difference = $actual_value - $expected_value;
					if (round($time_difference, 2) != 0) {
						// We're more than 10ms out - that's too much for being spec runtime
						return FALSE;
					}
				}
			}
			return TRUE;
		};
		return $matchers;
	}
}


class ViaApiCallNotificationWithFakableTime extends ViaApiCallNotification
{
	public function fakeStartMicrotimeAgo($time_ms)
	{
		$this->start_microtime = (microtime(TRUE) - ($time_ms / 1000));
	}
}
