<?php

namespace Red61\Via\Cache;

/**
 * Provides VIA caching using the PHP memcached extension
 *
 * This thin bridge stores all VIA data under a single memcached 'namespace' using the NamespacedMemcachedAdaptor.
 * This allows VIA data to be flushed separately from any other data in the cache, while allowing applications to use
 * the NamespacedMemcachedAdaptor directly for any of their own data without being coupled to VIA's ViaCacheDriver
 * interface.
 *
 * ### Using this driver with Red61_Via
 *
 * This driver will be automatically selected and configured if:
 *
 *  * Your integration uses the Red61_Via class to make API calls
 *  * You provide details of a memcached server backend in the Red61_Via options
 *  * You have the `memcached` extension installed
 *
 * In automatic mode, it will connect to a single server on the default memcached port (11211) with no custom options
 * and without using persistent connections. This should work "out of the box" for simple setups.
 *
 * For more complex setups, you should:
 *
 *   * follow the NamespacedMemcachedAdaptor documentation to create an instance of that class configured as required.
 *   * create an instance of this driver using your customised NamespacedMemcachedAdaptor
 *   * pass your driver instance in the `cache_handler` key of Red61_Via's `conf` array
 *
 */
class ViaMemcachedDriver implements ViaCacheDriver
{

    const CACHE_NAMESPACE = 'via';

    /**
     * @var NamespacedMemcachedAdaptor
     */
    protected $memcached_adaptor;

    /**
     * @param NamespacedMemcachedAdaptor $memcached_adaptor
     */
    public function __construct(NamespacedMemcachedAdaptor $memcached_adaptor)
    {
        $this->memcached_adaptor = $memcached_adaptor;
    }

    /**
     * {@inheritdoc}
     */
    public function lookup($namespace, $key, $ttl)
    {
        return $this->memcached_adaptor->get(self::CACHE_NAMESPACE, $key);
    }

    /**
     * {@inheritdoc}
     */
    public function store($namespace, $key, $value, $ttl)
    {
        $this->memcached_adaptor->set(self::CACHE_NAMESPACE, $key, $value, $ttl);
    }

    /**
     * {@inheritdoc}
     */
    public function flush()
    {
        $this->memcached_adaptor->flushNamespace(self::CACHE_NAMESPACE);
    }

} 
