<?php
/**
 * Defines apiCreateAccountRequestSpec - specifications for Red61\Via\ApiRequest\apiCreateAccountRequest
 *
 * @copyright  2014 Red61 Ltd
 * @licence    proprietary
 */

namespace spec\Red61\Via\ApiRequest;

use PhpSpec\Exception\Example\FailureException;
use Red61\Via\DataObject\ViaApiCustomerDetails;
use Red61\Via\DataObject\ViaApiCustomerOtherDetails;
use Red61\Via\Exception\ClientInvalidRequestException;
use Red61\Via\Plugin\ViaApiCallNotification;
use spec\ObjectBehavior;
use Prophecy\Argument;
use spec\Support\SubjectMethodFinder;

/**
 *
 * @see Red61\Via\ApiRequest\apiCreateAccountRequest
 */
class apiCreateAccountRequestSpec extends ObjectBehavior
{
    /**
     * Use $this->subject to get proper type hinting for the subject class
     * @var \Red61\Via\ApiRequest\apiCreateAccountRequest
     */
	protected $subject;

	function it_is_initializable()
    {
		$this->subject->shouldHaveType('Red61\Via\ApiRequest\apiCreateAccountRequest');
	}

	function it_is_a_preflight_filtering_request()
	{
		$this->subject->shouldHaveType('Red61\Via\ApiRequest\PreflightFilteringRequest');
	}

	function it_trims_preferred_contact_method()
	{
		$this->subject->setPreferredContactMethod('any ')
			->getPreferredContactMethod()->shouldBe('any');
	}

	function it_trims_forename()
	{
		$this->subject->setForename(' name ')
			->getForename()->shouldBe('name');
	}

	function it_trims_surname()
	{
		$this->subject->setSurname(' name ')
			->getSurname()->shouldBe('name');
	}

	function it_trims_email()
	{
		$this->subject->setSurname(' email@server.com ')
			->getSurname()->shouldBe('email@server.com');
	}

	/**
	 * @param \Red61\Via\Plugin\ViaApiCallNotification $notification
	 */
	function it_does_not_throw_preflight_with_valid_data($notification)
	{
		$this->given_set_with_valid_data_and(array());
		$this->subject->onBeforeViaCall($notification);
	}

	function it_throws_preflight_if_customer_id_not_set_and_create_false()
	{
		$this->given_set_with_valid_data_and(array('customerId' => NULL, 'create' => FALSE));
		$this->shouldThrowOnPreflightWithInvalidParameters(array('customerId'));
	}

	function it_throws_preflight_if_customer_id_set_and_create_true()
	{
		$this->given_set_with_valid_data_and(array('customerId' => 12, 'create' => TRUE));
		$this->shouldThrowOnPreflightWithInvalidParameters(array('customerId'));
	}

	function it_throws_preflight_if_customer_id_null_and_create_true()
	{
		$this->given_set_with_valid_data_and(array('customerId' => NULL, 'create' => TRUE));
		$this->shouldThrowOnPreflightWithInvalidParameters(array('customerId'));
	}

	function it_throws_preflight_if_surname_empty()
	{
		$this->given_set_with_valid_data_and(array('surname' => ''));
		$this->shouldThrowOnPreflightWithInvalidParameters(array('surname'));
	}

	function it_throws_preflight_if_email_not_set()
	{
		$this->given_set_with_valid_data_and(array('email' => ''));
		$this->shouldThrowOnPreflightWithInvalidParameters(array('email'));
	}

	function it_can_be_created_from_existing_customer_details_object()
	{
		$this->beConstructedThrough('createFromCustomer', array(new StubViaApiCustomerDetailsWithFieldNames));
		$this->subject->shouldBeAnInstanceOf('Red61\Via\ApiRequest\apiCreateAccountRequest');
	}

	function it_initialises_as_update_with_current_values_when_created_from_customer_details()
	{
		$this->beConstructedThrough('createFromCustomer', array(new StubViaApiCustomerDetailsWithFieldNames()));
		$this->subject->getCreate()->shouldBe(FALSE);
		$this->subject->getTitle()->shouldBe('title');
		$this->subject->getForename()->shouldBe('firstname');
		$this->subject->getSurname()->shouldBe('surname');
		$this->subject->getAddress1()->shouldBe('street1');
		$this->subject->getAddress2()->shouldBe('street2');
		$this->subject->getAddress3()->shouldBe('street3');
		$this->subject->getCity()->shouldBe('city');
		$this->subject->getCounty()->shouldBe('county');
		$this->subject->getPostcode()->shouldBe('postcode');
		$this->subject->getCountry()->shouldBe('country');
		$this->subject->getDayPhone()->shouldBe('other.dayphone');
		$this->subject->getEveningPhone()->shouldBe('other.evephone');
		$this->subject->getMobile()->shouldBe('other.mobile');
		$this->subject->getEmail()->shouldBe('other.email');
		$this->subject->getPreferredContactMethod()->shouldBe('other.prefcontact');
		$this->subject->getPassword()->shouldBe(NULL);
		$this->subject->getPasswordConfirm()->shouldBe(NULL);
		$this->subject->getOrganisationMailingPermission()->shouldBe('organisation_mailing');
		$this->subject->getThirdPartyMailingPermission()->shouldBe('thirdparty_mailing');
		$this->subject->getAdditionalMailingPermission()->shouldBe('additional_mailing');
		$this->subject->getCustomerId()->shouldBe('customerid');
		$this->subject->getWebKey()->shouldBe(NULL);
	}

	function it_initialises_all_request_fields_except_password_and_key_when_created_from_customer_details()
	{
		// Safety check against new properties being introduced

		$this->beConstructedThrough('createFromCustomer', array(new StubViaApiCustomerDetailsWithFieldNames()));
		foreach (SubjectMethodFinder::for_subject($this->subject)->find_matching('/^get/') as $getter) {
			if ( ! in_array($getter, array('getPassword', 'getPasswordConfirm', 'getWebKey'))) {
				$this->subject->$getter()->shouldNotBe(NULL);
			}
		}
	}

	/**
	 * Set up the object state
	 *
	 * @param mixed[] $properties
	 */
	protected function given_set_with_valid_data_and($properties)
	{
		$defaults = array(
			'create'     => TRUE,
			'customerId' => 0,
			'forename'   => 'first',
			'surname'    => 'last',
			'email'      => 'email'
		);
		$values = array_merge($defaults, $properties);
		$this->subject
			->setCreate($values['create'])
			->setCustomerId($values['customerId'])
			->setForename($values['forename'])
			->setSurname($values['surname'])
			->setEmail($values['email']);
	}

	/**
	 * @param string[] $expect_params
	 *
	 * @throws \PhpSpec\Exception\Example\FailureException
	 */
	protected function shouldThrowOnPreflightWithInvalidParameters($expect_params)
	{
		$notification = new ViaApiCallNotification($this->subject->getWrappedObject());
		try {
			$this->subject->onBeforeViaCall($notification);
			throw new FailureException("Expected exception was not thrown");
		} catch (ClientInvalidRequestException $e) {
			$invalid_params = $e->getInvalidParameters();
			foreach ($expect_params as $param_name) {
				expect($invalid_params)->toHaveKey($param_name);
			}
		}
	}

}

class StubViaApiCustomerDetailsWithFieldNames extends ViaApiCustomerDetails
{
	public function __construct()
	{
		$fields = array_keys(get_object_vars($this));
		foreach ($fields as $name) {
			$this->$name = $name;
		}
		$this->otherdetails = new StubViaApiCustomerOtherDetailsWithFieldNames;
	}
}

class StubViaApiCustomerOtherDetailsWithFieldNames extends ViaApiCustomerOtherDetails
{
	public function __construct()
	{
		$fields = array_keys(get_object_vars($this));
		foreach ($fields as $name) {
			$this->$name = 'other.'.$name;
		}
	}


}
